import { Injectable } from '@angular/core';
import moment from 'moment';

@Injectable({
  providedIn: 'root'
})
export class TimeDateService {

  constructor() { }

  getNext5WorkingDays(workingDaysList: string[]): string[] {
    const workingDays: any[] = [];
    const today = new Date();
    let count = 0; // Start from tomorrow

    while (workingDays.length < 5) {
      const nextDay = new Date(today);
      nextDay.setDate(today.getDate() + count);
      const dayOfWeek = nextDay.toLocaleString('en-US', { weekday: 'short' });
      const isWorkingDay = workingDaysList.includes(dayOfWeek);

      if (isWorkingDay) {
        const dayOfMonth = nextDay.getDate();
        const formattedDate = `${dayOfMonth < 10 ? '0' + dayOfMonth : dayOfMonth}`;
        workingDays.push(
          {
            name: `${formattedDate} ${dayOfWeek}`,
            value: this.removeTimeFromDate(nextDay).toDateString()
          }
        );
      }

      count++;
    }

    return workingDays;
  }



  getDayOfWeekFromString(dayString: string): number | null {
    const daysOfWeek: { [key: string]: number } = {
      'Sun': 0,
      'Mon': 1,
      'Tue': 2,
      'Wed': 3,
      'Thu': 4,
      'Fri': 5,
      'Sat': 6
    };

    return daysOfWeek[dayString] !== undefined ? daysOfWeek[dayString] : null;
  }

  getTimeSlots(startTime: string, endTime: string, breakStartTime: string, breakEndTime: string): string[] {
    const timeSlots: any[] = [];
    const start = new Date(`2024-01-01 ${startTime}`);
    const end = new Date(`2024-01-01 ${endTime}`);
    const breakStart = new Date(`2024-01-01 ${breakStartTime}`);
    const breakEnd = new Date(`2024-01-01 ${breakEndTime}`);
    const thirtyMinutes = 30 * 60 * 1000; // 30 minutes in milliseconds

    let currentTime = start.getTime();
    while (currentTime < end.getTime()) {
      const slotTime = new Date(currentTime);
      // Check if the slot falls within the break time range
      if (!(slotTime >= breakStart && slotTime < breakEnd)) {
        const hours = (slotTime.getHours() % 12 || 12).toString().padStart(2, '0'); // 12-hour format
        const period = slotTime.getHours() >= 12 ? 'PM' : 'AM'; // Get period (AM or PM)
        const minutes = slotTime.getMinutes().toString().padStart(2, '0');
        timeSlots.push(
          {
            name: `${hours}:${minutes} ${period}`,
            value: `${hours}:${minutes} ${period}`
          }
        );
      }
      currentTime += thirtyMinutes;
    }

    return timeSlots;
  }

  dateTimeToMilliseconds(date: Date, startTime: string): { startTime: number, endTime: number } {
    // Assuming time is in format 'h:mm A'
    const startDateTime = moment(date).set({ hour: moment(startTime, 'h:mm A').hours(), minute: moment(startTime, 'h:mm A').minutes() });
    const endDateTime = startDateTime.clone().add(30, 'minutes'); // Adding 30 minutes to start time

    const startTimeMillis = startDateTime.valueOf();
    const endTimeMillis = endDateTime.valueOf();

    return { startTime: startTimeMillis, endTime: endTimeMillis };
  }

  millisecondsToDateTime(milliseconds: number): { date: Date, time: string } {
    const dateTime =  new Date(milliseconds)
    // const date = new Date(dateTime.format('YYYY-MM-DD'));
    const date = this.removeTimeFromDate(moment(dateTime).toDate());
    const time = moment(dateTime).format('hh:mm A');
    const formattedDate = date.toLocaleDateString('en-US', { weekday: 'short', day: '2-digit' });
    return { date, time };
  }

  dateToMilliseconds(date: Date): number {
    return moment(date).valueOf();
  }

  removeTimeFromDate(date: Date): Date {
    const newDate = new Date(date);
    newDate.setHours(0, 0, 0, 0); // Set hours, minutes, seconds, and milliseconds to zero
    return newDate;
  }

  convertUTCToIST(utcMilliseconds: number): number {
    const utcDate = new Date(utcMilliseconds);

    // Get UTC offset in minutes
    const utcOffset = utcDate.getTimezoneOffset();

    // IST offset is 5 hours and 30 minutes ahead of UTC
    const istOffset = -330; // -5 hours * 60 minutes + -30 minutes

    // Convert UTC milliseconds to IST milliseconds
    const istMilliseconds = utcMilliseconds + (utcOffset + istOffset) * 60000;

    return istMilliseconds;
  }


}
