import { Component, EventEmitter, Input, Output } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { MatDialog } from '@angular/material/dialog';
import { MatSnackBar } from '@angular/material/snack-bar';
import { Router } from '@angular/router';
import { EducationService } from 'app/services/education.service';
import { TimeDateService } from 'app/services/time-date.service';
import moment from 'moment';
import { StatsComponent } from '../stats/stats.component';
import { MatTableDataSource } from '@angular/material/table';
import { SuccessDialogComponent } from 'app/components/controllers/success-dialog/success-dialog.component';
import { ValidationsService } from 'app/services/validations.service';
import { LoaderService } from 'app/services/loader.service';

@Component({
  selector: 'app-add-webinar',
  templateUrl: './add-webinar.component.html',
  styleUrls: ['./add-webinar.component.scss']
})
export class AddWebinarComponent {
  form: FormGroup;
  controlls: any;
  time = 0;
  @Input() editData: any;
  @Output() cancelEvent: EventEmitter<string> = new EventEmitter();
  constructor(
    private educationService: EducationService,
    private fb: FormBuilder,
    private router: Router,
    private snackBar: MatSnackBar,
    private timeDateService: TimeDateService,
    public dialog: MatDialog,
    private validationsService: ValidationsService,
    private loader: LoaderService
  ) {
    this.createForm();
    this.createControlls();
  }

  ngOnInit() {
    this.form.get('paidUnpaid').valueChanges.subscribe((change) => {
      this.form.get('price').reset()
      if (change == 'Paid') {
        this.form.get('price').setValidators([Validators.required]);
        this.form.get('price').updateValueAndValidity();

      } else {
        this.form.get('price').clearValidators();
        this.form.get('price').updateValueAndValidity();
      }
    })
  }
  ngOnChanges() {

    if (this.editData) {
      let ele = document.getElementById("add-update");
      ele.innerHTML = "Save";
      this.edit();
    }
  }

  onSetTime(t: number) {
    this.time = t
  }

  currentUTCDate(date): string {
    return moment.utc(date).local().format("DD-MMM-YYYY hh:mm A");
  }

  edit() {
    console.log(this.editData);
    this.form = this.fb.group({
      webinar: [this.editData['files'][0]['filePath'], Validators.required],
      title: [this.editData['title'], [Validators.required, this.validationsService.stringValidator()]],
      description: [this.editData['description'],Validators.required],
      poster: [this.editData['poster'], Validators.required],
      presenter: [this.editData['author'], [Validators.required, this.validationsService.stringValidator()]],
      publishedDate: [new Date(this.editData['publishDate'])],
      publishedTime: ['',],
      price: [this.editData['price'], [Validators.required, Validators.pattern('^[0-9]+$')]],
      liveDate: [this.timeDateService.millisecondsToDateTime(this.editData['liveDate'])['date'], Validators.required],
      liveTime: [this.timeDateService.millisecondsToDateTime(this.editData['liveDate'])['time'],Validators.required],
      publishedDateNdTime: [this.currentUTCDate(this.editData['publishDate'])],
      createdAt: [this.editData['createAt']],
      modifiedAt: [this.currentUTCDate(this.editData['modifiedAt'])],
      duration: [30, [Validators.pattern('^[0-9]+$')]],
      type: [this.editData['mode'], Validators.required],
      paidUnpaid: ['', Validators.required],
      featured: [this.editData['featured']],
      sponsored: [this.editData['sponsored']],
    })
    if (this.editData['paid'] == true) {
      this.form.get('paidUnpaid').setValue('Paid')
    } else {
      this.form.get('paidUnpaid').setValue('Free')
    }
  }

  createForm() {
    this.form = this.fb.group({
      webinar: ['', Validators.required],
      title: ['', [Validators.required, this.validationsService.stringValidator()]],
      description: ['',Validators.required],
      poster: ['', Validators.required],
      presenter: ['', [Validators.required, this.validationsService.stringValidator()]],
      publishedDate: [''],
      publishedTime: ['',],
      price: ['', [Validators.pattern('^[0-9]+$'), Validators.required]],
      liveDate: ['', Validators.required],
      liveTime: ['',Validators.required],
      publishedDateNdTime: [''],
      createdAt: [''],
      modifiedAt: [''],
      duration: ['',[Validators.required,Validators.pattern('^[0-9]+$')]],
      type: ['', Validators.required],
      paidUnpaid: ['', Validators.required],
      featured: [false],
      sponsored: [false],
    })
  }
  createControlls() {
    this.controlls = {
      webinar: {
        id: "webinar",
        name: 'webinar',
        label: "Upload Webinar*",
        required: true,
        type: "text"
      },
      publishedTime: {
        id: "publishedTime",
        name: "publishedTime",
        label: "Published Time"

      }, liveTime: {
        id: "liveTime",
        name: 'liveTime',
        label: "Live Time*",
        required: true,
      }, publishedDateNdTime: {
        id: "publishedDateNdTime",
        name: 'publishedDateNdTime',
        label: "Published Date and Time",
        required: false,
      },
      createdAt: {
        id: "createdAt",
        name: 'createdAt',
        label: "Created At",
        required: false,
      },
      modifiedAt: {
        id: "modifiedAt",
        name: 'modifiedAt',
        label: "Modified At",
        required: false,
      },
      title: {
        id: "title",
        name: 'title',
        label: "Title",
        required: true,
        placeholder: "Enter Title",
        type: "text"
      },
      price: {
        id: "price",
        name: 'price',
        label: "Price",
        required: true,
        placeholder: "Enter Price",
        type: "text"
      },
      description: {
        id: "description",
        name: 'description',
        label: "Description*",
        required: true,
        placeholder: "Enter Name",
        type: "richText"
      },
      poster: {
        id: "poster",
        name: 'poster',
        label: "Upload Webinar Poster*",
        required: true,
        type: "file"
      },
      presenter: {
        id: "presenter",
        name: 'presenter',
        label: "Presenter",
        required: true,
        type: "dropDown"
      },
      publishedDate: {
        id: "publishedDate",
        name: 'publishedDate',
        label: "Published Date",
        required: true,
        placeholder: "mm/dd/yyyy",
        type: "calender"
      },
      liveDate: {
        id: "liveDate",
        name: 'liveDate',
        label: "Live Date",
        required: true,
        placeholder: "mm/dd/yyyy",
        type: "calender"
      },
      duration: {
        id: "duration",
        name: 'duration',
        label: "Duration(in min)",
        required: true,
        placeholder: "0",
        type: "text"
      },
      type: {
        id: "type",
        name: 'type',
        label: "Type",
        required: true,
        type: "radio",
        options: [
          {
            name: "Online",
            value: "Online"
          }, {
            name: "Offline",
            value: "offline"
          }
        ]
      },
      paidUnpaid: {
        id: "paidUnpaid",
        name: 'paidUnpaid',
        label: "Service Type",
        required: true,
        type: "radio",
        options: [
          {
            name: "Free",
            value: "Free"
          }, {
            name: "Paid",
            value: "Paid"
          }
        ]
      },
      featured: {
        id: "featured",
        name: 'featured',
        label: "Featured",
        required: true,
        type: "boolean"
      },
      sponsored: {
        id: "sponsored",
        name: 'sponsored',
        label: "Sponsored",
        required: true,
        type: "boolean"
      },

    }
  }
  onNoClick(): void {
    this.cancelEvent.emit();
    this.router.navigate(['/education/webinar/']);
  }
  stats(type: string) {//stats
    console.log("stats");
    let stats: MatTableDataSource<any>
    let aray = []
    this.loader.showLoader();
    this.educationService.stats(this.editData['id'], 'EDU').subscribe((res: Array<any>) => {
      this.loader.hideLoader();
      res.forEach((el) => {
        if (el['viewed'] && type == 'views') {////liked shared
          console.log("viewed");
          aray.push(el)
          const statsArray = Object.values(aray);
          stats = new MatTableDataSource(statsArray.map((item: any) => ({ ...item, })));
        } else if (el['liked'] && type == 'likes') {
          console.log("liked");
          aray.push(el)
          const statsArray = Object.values(aray);
          stats = new MatTableDataSource(statsArray.map((item: any) => ({ ...item, })));

        } else if (el['shared'] && type == 'share') {
          console.log('shared');

          aray.push(el)
          const statsArray = Object.values(aray);
          stats = new MatTableDataSource(statsArray.map((item: any) => ({ ...item, })));

        }
      })
      const dialogRef = this.dialog.open(StatsComponent, {
        width: '600px',
        data: { stats: stats }
      });

    })

  }
  uploadWebinar(event) {
    console.log(event);
    if (this.editData) {
      this.loader.showLoader();
      this.educationService.updateWebinarVideo(this.editData['id'], this.editData['files'][0]['id'], this.form.get('webinar').value).subscribe((res) => {
        this.loader.hideLoader();
        console.log(res);
      })
    }
    // this.educationService.updateWebinarVideo()
  }

  uploadPoster(event) {
    if (this.editData) {
      this.loader.showLoader()
      this.educationService.uploadPoster(this.editData['id'], this.form.get('poster').value).subscribe((res) => {
        this.loader.hideLoader()
        console.log(res);
      })
    }
  }

  onSponsoredToggleChange(event) {
    console.log(event);

    if (this.editData) {
      this.educationService.featuresSponsoredTollge(this.editData['id'], 'sponsored', event).subscribe((res) => {
        console.log(res);
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Webinar edited successfully to sponsored! ', success: true },
          panelClass: 'success-dialog-container'
        });
      }, (error) => {
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Something Went Wrong!', success: false }
        });
      }
      )
    }
  }

  onFeatureToggleChange(event) {
    if (this.editData) {
      this.loader.showLoader()
      this.educationService.featuresSponsoredTollge(this.editData['id'], 'featured', event).subscribe((res) => {
        console.log(res);
        this.loader.hideLoader();
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Webinar edited successfully to featured!', success: true },
          panelClass: 'success-dialog-container'
        });
      }, (error) => {
        this.loader.hideLoader()
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Something Went Wrong!', success: false }
        });
      }
      )
    }
  }

  create() {
    let json = this.form.value
    console.log(json);
    // this.timeDateService.millisecondsToDateTime
    let body = {
      "title": json['title'],
      "description": json['description'],
      "eventOn": this.timeDateService.dateTimeToMilliseconds(json['liveDate'], json['liveTime'])['startTime'],
      "status": this.editData ? this.editData['status'] : "Inactive",
      "published": this.editData ? this.editData['status'] == 'Active' ? true : false : false,
      "mode": json['type'],
      "price": json['price'],
      "author": json['presenter'],
      "isPaid": json['paidUnpaid'],
      "eduCategory": "Webinar",
      "sponsored": this.editData ? this.editData['sponsored'] : json['sponsored'],//switchState
      "featured": this.editData ? this.editData['featured'] : json['featured'],
      // "publishDate": moment(json['publishedDate']).unix() * 1000, //json['publishedDate']
      "paid": false,
      "liveDate": this.timeDateService.dateTimeToMilliseconds(json['liveDate'], json['liveTime'])['startTime']//moment(json['liveDate']).unix() * 1000
    }
    if (json['paidUnpaid'] == "Paid") {
      body['paid'] = true
    } else {
      body['paid'] = false;
      body['price'] = 0
    }
    if (this.editData) {
      body['id'] = this.editData['id'];
      this.loader.showLoader();
      this.educationService.updateWebinar(body).subscribe((res) => {
        this.loader.hideLoader()
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Webinar updated successfully!', success: true },
          panelClass: 'success-dialog-container'
        });
        console.log(res);
        this.onNoClick();
      }, (error) => {
        this.loader.hideLoader();
        const dialogRef = this.dialog.open(SuccessDialogComponent, {
          width: '300px',
          data: { message: 'Something Went Wrong!', success: false }
        });
      }
      )
    } else {
      this.loader.showLoader();
      this.educationService.createWebiar(body).subscribe((res) => {
        this.educationService.uploadEpisode(res['id'], json['title'], json['description'], json['duration'], 0, json['webinar']).subscribe((res) => {
          this.router.navigate(['/education/webinar/']);
          console.log(res);
          const dialogRef = this.dialog.open(SuccessDialogComponent, {
            width: '300px',
            data: { message: 'Webinar created successfully!', success: true },
            panelClass: 'success-dialog-container'
          });
        }, (error) => {
          this.loader.hideLoader();
          const dialogRef = this.dialog.open(SuccessDialogComponent, {
            width: '300px',
            data: { message: 'Something Went Wrong!', success: false }
          });
        }
        )
        this.educationService.uploadPoster(res['id'], json['poster']).subscribe((res) => {
          this.loader.hideLoader()
          console.log(res);

        }, (error) => {
          this.loader.hideLoader();
        })
      })
    }
  }

  showSuccessToast(message: string): void {
    this.snackBar.open(message, 'Close', {
      duration: 3000,
      panelClass: ['success-toast'],
    });
  }

  showErrorToast(message: string): void {
    this.snackBar.open(message, 'Close', {
      duration: 3000,
      panelClass: ['error-toast'],
    });
  }
}
