import { DOCUMENT } from '@angular/common';
import { Injectable, Inject, Optional } from '@angular/core';
import { defer, firstValueFrom, isObservable } from 'rxjs';
import { shareReplay } from 'rxjs/operators';
import { defaultModules, QUILL_CONFIG_TOKEN, } from 'ngx-quill/config';
import * as i0 from "@angular/core";
class QuillService {
    constructor(injector, config) {
        this.config = config;
        this.quill$ = defer(async () => {
            if (!this.Quill) {
                // Quill adds events listeners on import https://github.com/quilljs/quill/blob/develop/core/emitter.js#L8
                // We'd want to use the unpatched `addEventListener` method to have all event callbacks to be run outside of zone.
                // We don't know yet if the `zone.js` is used or not, just save the value to restore it back further.
                const maybePatchedAddEventListener = this.document.addEventListener;
                // There're 2 types of Angular applications:
                // 1) zone-full (by default)
                // 2) zone-less
                // The developer can avoid importing the `zone.js` package and tells Angular that he/she is responsible for running
                // the change detection by himself. This is done by "nooping" the zone through `CompilerOptions` when bootstrapping
                // the root module. We fallback to `document.addEventListener` if `__zone_symbol__addEventListener` is not defined,
                // this means the `zone.js` is not imported.
                // The `__zone_symbol__addEventListener` is basically a native DOM API, which is not patched by zone.js, thus not even going
                // through the `zone.js` task lifecycle. You can also access the native DOM API as follows `target[Zone.__symbol__('methodName')]`.
                this.document.addEventListener =
                    // eslint-disable-next-line @typescript-eslint/dot-notation
                    this.document['__zone_symbol__addEventListener'] ||
                        this.document.addEventListener;
                const quillImport = await import(/* webpackChunkName: 'quill' */ 'quill');
                this.document.addEventListener = maybePatchedAddEventListener;
                this.Quill = (quillImport.default ? quillImport.default : quillImport);
            }
            // Only register custom options and modules once
            this.config.customOptions?.forEach((customOption) => {
                const newCustomOption = this.Quill.import(customOption.import);
                newCustomOption.whitelist = customOption.whitelist;
                this.Quill.register(newCustomOption, true, this.config.suppressGlobalRegisterWarning);
            });
            return await this.registerCustomModules(this.Quill, this.config.customModules, this.config.suppressGlobalRegisterWarning);
        }).pipe(shareReplay({ bufferSize: 1, refCount: true }));
        this.document = injector.get(DOCUMENT);
        if (!this.config) {
            this.config = { modules: defaultModules };
        }
    }
    getQuill() {
        return this.quill$;
    }
    /**
     * Marked as internal so it won't be available for `ngx-quill` consumers, this is only
     * internal method to be used within the library.
     *
     * @internal
     */
    async registerCustomModules(Quill, customModules, suppressGlobalRegisterWarning) {
        if (Array.isArray(customModules)) {
            // eslint-disable-next-line prefer-const
            for (let { implementation, path } of customModules) {
                // The `implementation` might be an observable that resolves the actual implementation,
                // e.g. if it should be lazy loaded.
                if (isObservable(implementation)) {
                    implementation = await firstValueFrom(implementation);
                }
                Quill.register(path, implementation, suppressGlobalRegisterWarning);
            }
        }
        // Return `Quill` constructor so we'll be able to re-use its return value except of using
        // `map` operators, etc.
        return Quill;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "16.2.8", ngImport: i0, type: QuillService, deps: [{ token: i0.Injector }, { token: QUILL_CONFIG_TOKEN, optional: true }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "16.2.8", ngImport: i0, type: QuillService, providedIn: 'root' }); }
}
export { QuillService };
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "16.2.8", ngImport: i0, type: QuillService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i0.Injector }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [QUILL_CONFIG_TOKEN]
                }] }]; } });
//# sourceMappingURL=data:application/json;base64,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