import * as i0 from '@angular/core';
import { Injectable, InjectionToken, Directive, Input, HostListener, Component, Pipe, Inject, Optional, EventEmitter, ViewChild, Output, ChangeDetectionStrategy, ContentChild, NgModule } from '@angular/core';
import * as i1 from '@angular/common';
import { DOCUMENT, CommonModule } from '@angular/common';
import { BehaviorSubject, Subject, merge } from 'rxjs';
import { shareReplay, tap, debounceTime, distinctUntilChanged, filter, takeUntil, map } from 'rxjs/operators';
import { trigger, transition, style, animate, sequence } from '@angular/animations';
import { DateTime, Info } from 'luxon';
import * as i4 from '@angular/forms';
import { FormControl, NG_VALUE_ACCESSOR, FormsModule, ReactiveFormsModule } from '@angular/forms';

var TimeUnit;
(function (TimeUnit) {
    TimeUnit[TimeUnit["HOUR"] = 0] = "HOUR";
    TimeUnit[TimeUnit["MINUTE"] = 1] = "MINUTE";
})(TimeUnit || (TimeUnit = {}));

var TimePeriod;
(function (TimePeriod) {
    TimePeriod["AM"] = "AM";
    TimePeriod["PM"] = "PM";
})(TimePeriod || (TimePeriod = {}));

var TimeFormat;
(function (TimeFormat) {
    TimeFormat["TWELVE"] = "hh:mm a";
    TimeFormat["TWELVE_SHORT"] = "h:m a";
    TimeFormat["TWENTY_FOUR"] = "HH:mm";
    TimeFormat["TWENTY_FOUR_SHORT"] = "H:m";
})(TimeFormat || (TimeFormat = {}));

function isSameOrAfter(time, compareWith, unit = 'minutes') {
    if (unit === 'hours') {
        return time.hour >= compareWith.hour;
    }
    if (unit === 'minutes') {
        return time.hasSame(compareWith, unit) || time.valueOf() > compareWith.valueOf();
    }
}
function isSameOrBefore(time, compareWith, unit = 'minutes') {
    if (unit === 'hours') {
        return time.hour <= compareWith.hour;
    }
    if (unit === 'minutes') {
        return time.hasSame(compareWith, unit) || time.valueOf() <= compareWith.valueOf();
    }
}
function isBetween(time, before, after, unit = 'minutes') {
    if (unit === 'hours') {
        return isSameOrBefore(time, after, unit) && isSameOrAfter(time, before, unit);
    }
    if (unit === 'minutes') {
        return isSameOrBefore(time, after) && isSameOrAfter(time, before);
    }
}
function isDigit(e) {
    // Allow: backspace, delete, tab, escape, enter
    if ([46, 8, 9, 27, 13].some(n => n === e.keyCode) ||
        // Allow: Ctrl/cmd+A
        (e.keyCode == 65 && (e.ctrlKey === true || e.metaKey === true)) ||
        // Allow: Ctrl/cmd+C
        (e.keyCode == 67 && (e.ctrlKey === true || e.metaKey === true)) ||
        // Allow: Ctrl/cmd+X
        (e.keyCode == 88 && (e.ctrlKey === true || e.metaKey === true)) ||
        // Allow: home, end, left, right, up, down
        (e.keyCode >= 35 && e.keyCode <= 40)) {
        return true;
    }
    return !((e.keyCode < 48 || e.keyCode > 57) && (e.keyCode < 96 || e.keyCode > 105));
}

// @dynamic
class TimeAdapter {
    static parseTime(time, opts) {
        const { numberingSystem, locale } = TimeAdapter.getLocaleOptionsByTime(time, opts);
        const isPeriodExist = time.split(' ').length === 2;
        const timeMask = isPeriodExist ? TimeFormat.TWELVE_SHORT : TimeFormat.TWENTY_FOUR_SHORT;
        return DateTime.fromFormat(time, timeMask, { numberingSystem, locale });
    }
    static formatTime(time, opts) {
        if (!time) {
            return 'Invalid Time';
        }
        const { format } = opts;
        const parsedTime = TimeAdapter.parseTime(time, opts).setLocale(TimeAdapter.DEFAULT_LOCALE);
        if (!parsedTime.isValid) {
            return null;
        }
        if (format !== 24) {
            return parsedTime.toLocaleString(Object.assign(Object.assign({}, DateTime.TIME_SIMPLE), { hour12: format !== 24, numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM })).replace(/\u200E/g, '').replace(/\u202F/g, ' ');
        }
        return parsedTime.toISOTime({
            includeOffset: false,
            suppressMilliseconds: true,
            suppressSeconds: true
        }).replace(/\u200E/g, '').replace(/\u202F/g, ' ');
    }
    static toLocaleTimeString(time, opts = {}) {
        const { format = TimeAdapter.DEFAULT_FORMAT, locale = TimeAdapter.DEFAULT_LOCALE } = opts;
        const hourCycle = format === 24 ? 'h23' : 'h12';
        const timeFormat = Object.assign(Object.assign({}, DateTime.TIME_SIMPLE), { hourCycle });
        const timeMask = (format === 24) ? TimeFormat.TWENTY_FOUR_SHORT : TimeFormat.TWELVE_SHORT;
        const localOpts = Object.assign({ locale: opts.locale, numberingSystem: opts.numberingSystem }, timeFormat);
        return DateTime.fromFormat(time, timeMask).setLocale(locale).toLocaleString(localOpts).replace(/\u202F/g, ' ');
    }
    static isTimeAvailable(time, min, max, granularity, minutesGap, format) {
        if (!time) {
            return;
        }
        const convertedTime = this.parseTime(time, { format });
        const minutes = convertedTime.minute;
        if (minutesGap && minutes === minutes && minutes % minutesGap !== 0) {
            throw new Error(`Your minutes - ${minutes} doesn\'t match your minutesGap - ${minutesGap}`);
        }
        const isAfter = (min && !max)
            && isSameOrAfter(convertedTime, min, granularity);
        const isBefore = (max && !min)
            && isSameOrBefore(convertedTime, max, granularity);
        const between = (min && max)
            && isBetween(convertedTime, min, max, granularity);
        const isAvailable = !min && !max;
        return isAfter || isBefore || between || isAvailable;
    }
    /***
     *  Format hour according to time format (12 or 24)
     */
    static formatHour(currentHour, format, period) {
        if (format === 24) {
            return currentHour;
        }
        const hour = period === TimePeriod.AM ? currentHour : currentHour + 12;
        if (period === TimePeriod.AM && hour === 12) {
            return 0;
        }
        else if (period === TimePeriod.PM && hour === 24) {
            return 12;
        }
        return hour;
    }
    static fromDateTimeToString(time, format) {
        const timeFormat = format === 24 ? TimeFormat.TWENTY_FOUR : TimeFormat.TWELVE;
        return time.reconfigure({
            numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM,
            locale: TimeAdapter.DEFAULT_LOCALE
        }).toFormat(timeFormat).replace(/\u202F/g, ' ');
    }
    static getLocaleOptionsByTime(time, opts) {
        const localeConfig = { numberingSystem: opts.numberingSystem, locale: opts.locale };
        const defaultConfig = { numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM, locale: TimeAdapter.DEFAULT_LOCALE };
        return isNaN(parseInt(time, 10)) ? localeConfig : defaultConfig;
    }
}
TimeAdapter.DEFAULT_FORMAT = 12;
TimeAdapter.DEFAULT_LOCALE = 'en-US';
TimeAdapter.DEFAULT_NUMBERING_SYSTEM = 'latn';

const DEFAULT_HOUR = {
    time: 12,
    angle: 360
};
const DEFAULT_MINUTE = {
    time: 0,
    angle: 360
};
class NgxMaterialTimepickerService {
    constructor() {
        this.hourSubject = new BehaviorSubject(DEFAULT_HOUR);
        this.minuteSubject = new BehaviorSubject(DEFAULT_MINUTE);
        this.periodSubject = new BehaviorSubject(TimePeriod.AM);
    }
    set hour(hour) {
        this.hourSubject.next(hour);
    }
    get selectedHour() {
        return this.hourSubject.asObservable();
    }
    set minute(minute) {
        this.minuteSubject.next(minute);
    }
    get selectedMinute() {
        return this.minuteSubject.asObservable();
    }
    set period(period) {
        const isPeriodValid = (period === TimePeriod.AM) || (period === TimePeriod.PM);
        if (isPeriodValid) {
            this.periodSubject.next(period);
        }
    }
    get selectedPeriod() {
        return this.periodSubject.asObservable();
    }
    setDefaultTimeIfAvailable(time, min, max, format, minutesGap) {
        /* Workaround to double error message*/
        try {
            if (TimeAdapter.isTimeAvailable(time, min, max, 'minutes', minutesGap)) {
                this.setDefaultTime(time, format);
            }
        }
        catch (e) {
            console.error(e);
        }
    }
    getFullTime(format) {
        const selectedHour = this.hourSubject.getValue().time;
        const selectedMinute = this.minuteSubject.getValue().time;
        const hour = selectedHour != null ? selectedHour : DEFAULT_HOUR.time;
        const minute = selectedMinute != null ? selectedMinute : DEFAULT_MINUTE.time;
        const period = format === 12 ? this.periodSubject.getValue() : '';
        const time = `${hour}:${minute} ${period}`.trim();
        return TimeAdapter.formatTime(time, { format });
    }
    setDefaultTime(time, format) {
        const defaultTime = TimeAdapter.parseTime(time, { format }).toJSDate();
        if (DateTime.fromJSDate(defaultTime).isValid) {
            const period = time.substr(time.length - 2).toUpperCase();
            const hour = defaultTime.getHours();
            this.hour = Object.assign(Object.assign({}, DEFAULT_HOUR), { time: formatHourByPeriod(hour, period) });
            this.minute = Object.assign(Object.assign({}, DEFAULT_MINUTE), { time: defaultTime.getMinutes() });
            this.period = period;
        }
        else {
            this.resetTime();
        }
    }
    resetTime() {
        this.hour = Object.assign({}, DEFAULT_HOUR);
        this.minute = Object.assign({}, DEFAULT_MINUTE);
        this.period = TimePeriod.AM;
    }
}
NgxMaterialTimepickerService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, deps: [], target: i0.ɵɵFactoryTarget.Injectable });
NgxMaterialTimepickerService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root'
                }]
        }] });
/***
 *  Format hour in 24hours format to meridian (AM or PM) format
 */
function formatHourByPeriod(hour, period) {
    switch (period) {
        case TimePeriod.AM:
            return hour === 0 ? 12 : hour;
        case TimePeriod.PM:
            return hour === 12 ? 12 : hour - 12;
        default:
            return hour;
    }
}

const TIME_LOCALE = new InjectionToken('TimeLocale', {
    providedIn: 'root',
    factory: () => TimeAdapter.DEFAULT_LOCALE
});
const NUMBERING_SYSTEM = new InjectionToken('NumberingSystem', {
    providedIn: 'root',
    factory: () => TimeAdapter.DEFAULT_NUMBERING_SYSTEM
});

class NgxMaterialTimepickerEventService {
    constructor() {
        this.backdropClickSubject = new Subject();
        this.keydownEventSubject = new Subject();
    }
    get backdropClick() {
        return this.backdropClickSubject.asObservable().pipe(shareReplay({ bufferSize: 1, refCount: true }));
    }
    get keydownEvent() {
        return this.keydownEventSubject.asObservable().pipe(shareReplay({ bufferSize: 1, refCount: true }));
    }
    dispatchEvent(event) {
        switch (event.type) {
            case 'click':
                this.backdropClickSubject.next(event);
                break;
            case 'keydown':
                this.keydownEventSubject.next(event);
                break;
            default:
                throw new Error('no such event type');
        }
    }
}
NgxMaterialTimepickerEventService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerEventService, deps: [], target: i0.ɵɵFactoryTarget.Injectable });
NgxMaterialTimepickerEventService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerEventService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerEventService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root'
                }]
        }] });

class AppendToInputDirective {
    constructor(elementRef, renderer) {
        this.renderer = renderer;
        this.element = elementRef.nativeElement;
    }
    get inputCords() {
        return this.inputElement.getBoundingClientRect();
    }
    get direction() {
        const height = this.element.offsetHeight;
        const { bottom, top } = this._inputCords;
        const isElementFit = (window && window.innerHeight) - bottom < height;
        const isTop = isElementFit && top > height;
        const isCenter = isElementFit && top < height;
        if (isTop) {
            return 'top';
        }
        else if (isCenter) {
            return 'center';
        }
        return 'bottom';
    }
    ngAfterViewInit() {
        this._inputCords = this.inputCords;
        this._direction = this.direction;
        this.append();
    }
    changePosition() {
        const { bottom, top } = this.inputCords;
        const y = this.defineElementYByDirection(top, bottom);
        this.setStyle('top', `${y}px`);
    }
    append() {
        const { left, bottom, top } = this._inputCords;
        const y = this.defineElementYByDirection(top, bottom);
        this.setStyle('position', 'fixed');
        this.setStyle('left', `${left}px`);
        this.setStyle('top', `${y}px`);
    }
    setStyle(style, value) {
        this.renderer.setStyle(this.element, style, value);
    }
    defineElementYByDirection(inputTop, inputBottom) {
        if (this._direction === 'top') {
            return inputTop - this.element.offsetHeight;
        }
        else if (this._direction === 'center') {
            return inputTop - (this.element.offsetHeight / 2);
        }
        return inputBottom;
    }
}
AppendToInputDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: AppendToInputDirective, deps: [{ token: i0.ElementRef }, { token: i0.Renderer2 }], target: i0.ɵɵFactoryTarget.Directive });
AppendToInputDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: AppendToInputDirective, selector: "[ngxAppendToInput]", inputs: { inputElement: ["ngxAppendToInput", "inputElement"] }, host: { listeners: { "window:scroll": "changePosition()" } }, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: AppendToInputDirective, decorators: [{
            type: Directive,
            args: [{
                    selector: '[ngxAppendToInput]'
                }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: i0.Renderer2 }]; }, propDecorators: { inputElement: [{
                type: Input,
                args: ['ngxAppendToInput']
            }], changePosition: [{
                type: HostListener,
                args: ['window:scroll']
            }] } });

class NgxMaterialTimepickerContentComponent {
}
NgxMaterialTimepickerContentComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerContentComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerContentComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerContentComponent, selector: "ngx-material-timepicker-content", inputs: { appendToInput: "appendToInput", inputElement: "inputElement" }, ngImport: i0, template: "<div [ngxAppendToInput]=\"inputElement\" *ngIf=\"appendToInput;else timepickerModal\">\n    <!--suppress HtmlUnknownAttribute -->\n    <ng-container *ngTemplateOutlet=\"timepickerOutlet\"></ng-container>\n</div>\n\n<ng-template #timepickerModal>\n    <!--suppress HtmlUnknownAttribute -->\n    <ng-container *ngTemplateOutlet=\"timepickerOutlet\"></ng-container>\n</ng-template>\n\n<ng-template #timepickerOutlet>\n    <ng-content></ng-content>\n</ng-template>\n", directives: [{ type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: AppendToInputDirective, selector: "[ngxAppendToInput]", inputs: ["ngxAppendToInput"] }, { type: i1.NgTemplateOutlet, selector: "[ngTemplateOutlet]", inputs: ["ngTemplateOutletContext", "ngTemplateOutlet"] }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerContentComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-content',
                    templateUrl: './ngx-material-timepicker-content.component.html',
                }]
        }], propDecorators: { appendToInput: [{
                type: Input
            }], inputElement: [{
                type: Input
            }] } });

// @dynamic
class TimepickerTimeUtils {
    static getHours(format) {
        return Array(format).fill(1).map((v, i) => {
            const angleStep = 30;
            const time = v + i;
            const angle = angleStep * time;
            return { time: time === 24 ? 0 : time, angle };
        });
    }
    static disableHours(hours, config) {
        if (config.min || config.max) {
            return hours.map(value => {
                const hour = config.format === 24 ? value.time : TimeAdapter.formatHour(value.time, config.format, config.period);
                const currentTime = DateTime.fromObject({ hour }).toFormat(TimeFormat.TWELVE);
                return Object.assign(Object.assign({}, value), { disabled: !TimeAdapter.isTimeAvailable(currentTime, config.min, config.max, 'hours') });
            });
        }
        return hours;
    }
    static getMinutes(gap = 1) {
        const minutesCount = 60;
        const angleStep = 360 / minutesCount;
        const minutes = [];
        for (let i = 0; i < minutesCount; i++) {
            const angle = angleStep * i;
            if (i % gap === 0) {
                minutes.push({ time: i, angle: angle !== 0 ? angle : 360 });
            }
        }
        return minutes;
    }
    static disableMinutes(minutes, selectedHour, config) {
        if (config.min || config.max) {
            const hour = TimeAdapter.formatHour(selectedHour, config.format, config.period);
            return minutes.map(value => {
                const currentTime = DateTime.fromObject({ hour, minute: value.time }).toFormat(TimeFormat.TWELVE);
                return Object.assign(Object.assign({}, value), { disabled: !TimeAdapter.isTimeAvailable(currentTime, config.min, config.max, 'minutes') });
            });
        }
        return minutes;
    }
}

class TimeLocalizerPipe {
    constructor(locale) {
        this.locale = locale;
    }
    transform(time, timeUnit, isKeyboardEnabled = false) {
        if (time == null || time === '') {
            return '';
        }
        switch (timeUnit) {
            case TimeUnit.HOUR: {
                const format = (time === 0 || isKeyboardEnabled) ? 'HH' : 'H';
                return this.formatTime('hour', time, format);
            }
            case TimeUnit.MINUTE:
                return this.formatTime('minute', time, 'mm');
            default:
                throw new Error(`There is no Time Unit with type ${timeUnit}`);
        }
    }
    formatTime(timeMeasure, time, format) {
        try {
            return DateTime.fromObject({ [timeMeasure]: +time }).setLocale(this.locale).toFormat(format);
        }
        catch (_a) {
            throw new Error(`Cannot format provided time - ${time} to locale - ${this.locale}`);
        }
    }
}
TimeLocalizerPipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeLocalizerPipe, deps: [{ token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Pipe });
TimeLocalizerPipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeLocalizerPipe, name: "timeLocalizer" });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeLocalizerPipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'timeLocalizer'
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; } });

class TimeParserPipe {
    constructor(locale, numberingSystem) {
        this.locale = locale;
        this.numberingSystem = numberingSystem;
    }
    transform(time, timeUnit = TimeUnit.HOUR) {
        if (time == null || time === '') {
            return '';
        }
        if (!isNaN(+time)) {
            return time;
        }
        if (timeUnit === TimeUnit.MINUTE) {
            return this.parseTime(time, 'm', 'minute');
        }
        return this.parseTime(time, 'H', 'hour');
    }
    parseTime(time, format, timeMeasure) {
        const parsedTime = DateTime.fromFormat(String(time), format, {
            numberingSystem: this.numberingSystem,
            locale: this.locale
        })[timeMeasure];
        if (!isNaN(parsedTime)) {
            return parsedTime;
        }
        throw new Error(`Cannot parse time - ${time}`);
    }
}
TimeParserPipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeParserPipe, deps: [{ token: TIME_LOCALE }, { token: NUMBERING_SYSTEM }], target: i0.ɵɵFactoryTarget.Pipe });
TimeParserPipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeParserPipe, name: "timeParser" });
TimeParserPipe.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeParserPipe });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeParserPipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'timeParser'
                }]
        }, {
            type: Injectable
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [NUMBERING_SYSTEM]
                }] }]; } });

class AutofocusDirective {
    constructor(element, document) {
        this.element = element;
        this.document = document;
        this.activeElement = this.document.activeElement;
    }
    ngOnChanges() {
        if (this.isFocusActive) {
            // To avoid ExpressionChangedAfterItHasBeenCheckedError;
            setTimeout(() => this.element.nativeElement.focus({ preventScroll: true }));
        }
    }
    ngOnDestroy() {
        // To avoid ExpressionChangedAfterItHasBeenCheckedError;
        setTimeout(() => this.activeElement.focus({ preventScroll: true }));
    }
}
AutofocusDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: AutofocusDirective, deps: [{ token: i0.ElementRef }, { token: DOCUMENT, optional: true }], target: i0.ɵɵFactoryTarget.Directive });
AutofocusDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: AutofocusDirective, selector: "[timepickerAutofocus]", inputs: { isFocusActive: ["timepickerAutofocus", "isFocusActive"] }, usesOnChanges: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: AutofocusDirective, decorators: [{
            type: Directive,
            args: [{
                    selector: '[timepickerAutofocus]'
                }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }]; }, propDecorators: { isFocusActive: [{
                type: Input,
                args: ['timepickerAutofocus']
            }] } });

/* tslint:disable:triple-equals */
class NgxMaterialTimepickerDialControlComponent {
    constructor(timeParserPipe, timeLocalizerPipe) {
        this.timeParserPipe = timeParserPipe;
        this.timeLocalizerPipe = timeLocalizerPipe;
        this.timeUnitChanged = new EventEmitter();
        this.timeChanged = new EventEmitter();
        this.focused = new EventEmitter();
        this.unfocused = new EventEmitter();
    }
    get selectedTime() {
        if (!!this.time) {
            return this.timeList.find(t => t.time === +this.time);
        }
    }
    ngOnInit() {
        if (this.isEditable) {
            this.timeControl = new FormControl({ value: this.formatTimeForUI(this.time), disabled: this.disabled });
            this.timeControl.valueChanges.pipe(tap((value) => {
                if (value.length > 2) {
                    this.updateInputValue(value.slice(-1));
                }
            }), debounceTime(500), distinctUntilChanged(), filter((value) => !isTimeDisabledToChange(this.time, value, this.timeList)), tap((value) => this.time = this.timeParserPipe.transform(value, this.timeUnit).toString())).subscribe(() => this.updateTime());
        }
    }
    saveTimeAndChangeTimeUnit(event, unit) {
        event.preventDefault();
        this.previousTime = this.time;
        this.timeUnitChanged.next(unit);
        this.focused.next();
    }
    updateTime() {
        const time = this.selectedTime;
        if (time) {
            this.timeChanged.next(time);
            this.previousTime = time.time;
            if (this.isEditable) {
                this.updateInputValue(this.formatTimeForUI(time.time));
            }
        }
    }
    onKeydown(e) {
        if (!isDigit(e)) {
            e.preventDefault();
        }
        else {
            this.changeTimeByArrow(e.keyCode);
        }
    }
    changeTimeByArrow(keyCode) {
        const ARROW_UP = 38;
        const ARROW_DOWN = 40;
        let time;
        if (keyCode === ARROW_UP) {
            time = String(+this.time + (this.minutesGap || 1));
        }
        else if (keyCode === ARROW_DOWN) {
            time = String(+this.time - (this.minutesGap || 1));
        }
        if (!isTimeUnavailable(time, this.timeList)) {
            this.time = time;
            this.updateTime();
        }
    }
    formatTimeForUI(value) {
        const parsedTime = this.timeParserPipe.transform(value, this.timeUnit).toString();
        return this.timeLocalizerPipe.transform(parsedTime, this.timeUnit, true);
    }
    updateInputValue(value) {
        this.editableTimeTmpl.nativeElement.value = value;
    }
}
NgxMaterialTimepickerDialControlComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerDialControlComponent, deps: [{ token: TimeParserPipe }, { token: TimeLocalizerPipe }], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerDialControlComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerDialControlComponent, selector: "ngx-material-timepicker-dial-control", inputs: { timeList: "timeList", timeUnit: "timeUnit", time: "time", isActive: "isActive", isEditable: "isEditable", minutesGap: "minutesGap", disabled: "disabled" }, outputs: { timeUnitChanged: "timeUnitChanged", timeChanged: "timeChanged", focused: "focused", unfocused: "unfocused" }, providers: [TimeParserPipe, TimeLocalizerPipe], viewQueries: [{ propertyName: "editableTimeTmpl", first: true, predicate: ["editableTimeTmpl"], descendants: true }], ngImport: i0, template: "<!--suppress HtmlFormInputWithoutLabel, HtmlUnknownAttribute -->\n<input class=\"timepicker-dial__control timepicker-dial__item\"\n       [ngClass]=\"{'timepicker-dial__item_active': isActive}\"\n       [ngModel]=\"time | timeLocalizer: timeUnit\"\n       (ngModelChange)=\"time = $event\"\n       [disabled]=\"disabled\"\n       (input)=\"updateTime()\" (focus)=\"saveTimeAndChangeTimeUnit($event, timeUnit)\"\n       readonly [timepickerAutofocus]=\"isActive\"\n       *ngIf=\"!isEditable;else editableTemplate\">\n\n<ng-template #editableTemplate>\n    <!--suppress HtmlFormInputWithoutLabel, HtmlUnknownAttribute -->\n    <input class=\"timepicker-dial__control timepicker-dial__item timepicker-dial__control_editable\"\n           #editableTimeTmpl\n           [formControl]=\"timeControl\"\n           [ngClass]=\"{'timepicker-dial__item_active': isActive}\"\n           [timepickerAutofocus]=\"isActive\"\n           (focus)=\"saveTimeAndChangeTimeUnit($event, timeUnit)\"\n           (keydown)=\"onKeydown($event)\">\n</ng-template>\n", styles: [".timepicker-dial__item{cursor:pointer;color:#ffffff80;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-dial__item{font-family:var(--primary-font-family);color:var(--dial-inactive-color)}}.timepicker-dial__item_active{color:#fff}@supports (color: var(--dial-active-color)){.timepicker-dial__item_active{color:var(--dial-active-color)}}.timepicker-dial__control{border:none;background-color:transparent;font-size:50px;width:60px;padding:0;border-radius:3px;text-align:right}.timepicker-dial__control_editable:focus{color:#00bfff;background-color:#fff;outline:deepskyblue}@supports (color: var(--dial-editable-active-color)){.timepicker-dial__control_editable:focus{color:var(--dial-editable-active-color)}}@supports (background-color: var(--dial-editable-background-color)){.timepicker-dial__control_editable:focus{background-color:var(--dial-editable-background-color)}}@supports (outline: var(--dial-editable-active-color)){.timepicker-dial__control_editable:focus{outline:var(--dial-editable-active-color)}}.timepicker-dial__control:disabled{cursor:default}.timepicker-dial__control:focus-visible{outline:none}\n"], directives: [{ type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: i4.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i4.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { type: i4.NgModel, selector: "[ngModel]:not([formControlName]):not([formControl])", inputs: ["name", "disabled", "ngModel", "ngModelOptions"], outputs: ["ngModelChange"], exportAs: ["ngModel"] }, { type: AutofocusDirective, selector: "[timepickerAutofocus]", inputs: ["timepickerAutofocus"] }, { type: i4.FormControlDirective, selector: "[formControl]", inputs: ["disabled", "formControl", "ngModel"], outputs: ["ngModelChange"], exportAs: ["ngForm"] }], pipes: { "timeLocalizer": TimeLocalizerPipe } });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerDialControlComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-dial-control',
                    templateUrl: 'ngx-material-timepicker-dial-control.component.html',
                    styleUrls: ['ngx-material-timepicker-dial-control.component.scss'],
                    providers: [TimeParserPipe, TimeLocalizerPipe],
                }]
        }], ctorParameters: function () { return [{ type: TimeParserPipe }, { type: TimeLocalizerPipe }]; }, propDecorators: { timeList: [{
                type: Input
            }], timeUnit: [{
                type: Input
            }], time: [{
                type: Input
            }], isActive: [{
                type: Input
            }], isEditable: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], disabled: [{
                type: Input
            }], editableTimeTmpl: [{
                type: ViewChild,
                args: ['editableTimeTmpl']
            }], timeUnitChanged: [{
                type: Output
            }], timeChanged: [{
                type: Output
            }], focused: [{
                type: Output
            }], unfocused: [{
                type: Output
            }] } });
function isTimeDisabledToChange(currentTime, nextTime, timeList) {
    const isNumber = /\d/.test(nextTime);
    if (isNumber) {
        return isTimeUnavailable(nextTime, timeList);
    }
}
function isTimeUnavailable(time, timeList) {
    const selectedTime = timeList.find(value => value.time === +time);
    return !selectedTime || (selectedTime && selectedTime.disabled);
}

class NgxMaterialTimepickerPeriodComponent {
    constructor() {
        this.timePeriod = TimePeriod;
        this.isPeriodAvailable = true;
        this.periodChanged = new EventEmitter();
    }
    changePeriod(period) {
        this.isPeriodAvailable = this.isSwitchPeriodAvailable(period);
        if (this.isPeriodAvailable) {
            this.periodChanged.next(period);
        }
    }
    animationDone() {
        this.isPeriodAvailable = true;
    }
    isSwitchPeriodAvailable(period) {
        const time = this.getDisabledTimeByPeriod(period);
        return !time.every(t => t.disabled);
    }
    getDisabledTimeByPeriod(period) {
        switch (this.activeTimeUnit) {
            case TimeUnit.HOUR:
                return TimepickerTimeUtils.disableHours(this.hours, {
                    min: this.minTime,
                    max: this.maxTime,
                    format: this.format,
                    period
                });
            case TimeUnit.MINUTE:
                return TimepickerTimeUtils.disableMinutes(this.minutes, +this.selectedHour, {
                    min: this.minTime,
                    max: this.maxTime,
                    format: this.format,
                    period
                });
            default:
                throw new Error('no such TimeUnit');
        }
    }
}
NgxMaterialTimepickerPeriodComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerPeriodComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerPeriodComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerPeriodComponent, selector: "ngx-material-timepicker-period", inputs: { selectedPeriod: "selectedPeriod", format: "format", activeTimeUnit: "activeTimeUnit", hours: "hours", minutes: "minutes", minTime: "minTime", maxTime: "maxTime", selectedHour: "selectedHour", meridiems: "meridiems" }, outputs: { periodChanged: "periodChanged" }, ngImport: i0, template: "<div class=\"timepicker-period\">\n\t\t\t<button class=\"timepicker-dial__item timepicker-period__btn\"\n                  [ngClass]=\"{'timepicker-dial__item_active': selectedPeriod === timePeriod.AM}\"\n                  (click)=\"changePeriod(timePeriod.AM)\"\n                  type=\"button\">{{meridiems[0]}}</button>\n    <button class=\"timepicker-dial__item timepicker-period__btn\"\n          [ngClass]=\"{'timepicker-dial__item_active': selectedPeriod === timePeriod.PM}\"\n          (click)=\"changePeriod(timePeriod.PM)\"\n          type=\"button\">{{meridiems[1]}}</button>\n    <div class=\"timepicker-period__warning\" [@scaleInOut] (@scaleInOut.done)=\"animationDone()\" *ngIf=\"!isPeriodAvailable\">\n        <p>Current time would be invalid in this period.</p>\n    </div>\n</div>\n", styles: [".timepicker-dial__item{cursor:pointer;color:#ffffff80;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-dial__item{font-family:var(--primary-font-family);color:var(--dial-inactive-color)}}.timepicker-dial__item_active{color:#fff}@supports (color: var(--dial-active-color)){.timepicker-dial__item_active{color:var(--dial-active-color)}}.timepicker-period{display:flex;flex-direction:column;position:relative}.timepicker-period__btn{padding:1px 3px;border:0;background-color:transparent;font-size:18px;font-weight:500;-webkit-user-select:none;-moz-user-select:none;user-select:none;outline:none;border-radius:3px;transition:background-color .5s;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-period__btn{font-family:var(--primary-font-family)}}.timepicker-period__btn:focus{background-color:#00000012}.timepicker-period__warning{padding:5px 10px;border-radius:3px;background-color:#0000008c;color:#fff;position:absolute;width:200px;left:-20px;top:40px}.timepicker-period__warning>p{margin:0;font-size:12px;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-period__warning>p{font-family:var(--primary-font-family)}}\n"], directives: [{ type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }], animations: [
        trigger('scaleInOut', [
            transition(':enter', [
                style({ transform: 'scale(0)' }),
                animate('.2s', style({ transform: 'scale(1)' })),
                sequence([
                    animate('3s', style({ opacity: 1 })),
                    animate('.3s', style({ opacity: 0 }))
                ])
            ])
        ])
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerPeriodComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-period',
                    templateUrl: 'ngx-material-timepicker-period.component.html',
                    styleUrls: ['ngx-material-timepicker-period.component.scss'],
                    animations: [
                        trigger('scaleInOut', [
                            transition(':enter', [
                                style({ transform: 'scale(0)' }),
                                animate('.2s', style({ transform: 'scale(1)' })),
                                sequence([
                                    animate('3s', style({ opacity: 1 })),
                                    animate('.3s', style({ opacity: 0 }))
                                ])
                            ])
                        ])
                    ]
                }]
        }], propDecorators: { selectedPeriod: [{
                type: Input
            }], format: [{
                type: Input
            }], activeTimeUnit: [{
                type: Input
            }], hours: [{
                type: Input
            }], minutes: [{
                type: Input
            }], minTime: [{
                type: Input
            }], maxTime: [{
                type: Input
            }], selectedHour: [{
                type: Input
            }], meridiems: [{
                type: Input
            }], periodChanged: [{
                type: Output
            }] } });

class NgxMaterialTimepickerDialComponent {
    constructor(locale) {
        this.locale = locale;
        this.timeUnit = TimeUnit;
        this.meridiems = Info.meridiems({ locale: this.locale });
        this.periodChanged = new EventEmitter();
        this.timeUnitChanged = new EventEmitter();
        this.hourChanged = new EventEmitter();
        this.minuteChanged = new EventEmitter();
    }
    ngOnChanges(changes) {
        if (changes['period'] && changes['period'].currentValue
            || changes['format'] && changes['format'].currentValue) {
            const hours = TimepickerTimeUtils.getHours(this.format);
            this.hours = TimepickerTimeUtils.disableHours(hours, {
                min: this.minTime,
                max: this.maxTime,
                format: this.format,
                period: this.period,
            });
        }
        if (changes['period'] && changes['period'].currentValue
            || changes['hour'] && changes['hour'].currentValue) {
            const minutes = TimepickerTimeUtils.getMinutes(this.minutesGap);
            this.minutes = TimepickerTimeUtils.disableMinutes(minutes, +this.hour, {
                min: this.minTime,
                max: this.maxTime,
                format: this.format,
                period: this.period,
            });
        }
    }
    changeTimeUnit(unit) {
        this.timeUnitChanged.next(unit);
    }
    changePeriod(period) {
        this.periodChanged.next(period);
    }
    changeHour(hour) {
        this.hourChanged.next(hour);
        if (this.isEditable) {
            this.changeTimeUnit(TimeUnit.MINUTE);
        }
    }
    changeMinute(minute) {
        this.minuteChanged.next(minute);
    }
    showHint() {
        this.isHintVisible = true;
    }
    hideHint() {
        this.isHintVisible = false;
    }
}
NgxMaterialTimepickerDialComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerDialComponent, deps: [{ token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerDialComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerDialComponent, selector: "ngx-material-timepicker-dial", inputs: { editableHintTmpl: "editableHintTmpl", hour: "hour", minute: "minute", format: "format", period: "period", activeTimeUnit: "activeTimeUnit", minTime: "minTime", maxTime: "maxTime", isEditable: "isEditable", minutesGap: "minutesGap", hoursOnly: "hoursOnly" }, outputs: { periodChanged: "periodChanged", timeUnitChanged: "timeUnitChanged", hourChanged: "hourChanged", minuteChanged: "minuteChanged" }, usesOnChanges: true, ngImport: i0, template: "<div class=\"timepicker-dial\">\n    <div class=\"timepicker-dial__container\">\n        <div class=\"timepicker-dial__time\">\n            <ngx-material-timepicker-dial-control [timeList]=\"hours\" [time]=\"hour\" [timeUnit]=\"timeUnit.HOUR\"\n                                                  [isActive]=\"activeTimeUnit === timeUnit.HOUR\"\n                                                  [isEditable]=\"isEditable\"\n                                                  (timeUnitChanged)=\"changeTimeUnit($event)\"\n                                                  (timeChanged)=\"changeHour($event)\"\n                                                  (focused)=\"showHint()\"\n                                                  (unfocused)=\"hideHint()\">\n\n            </ngx-material-timepicker-dial-control>\n            <span>:</span>\n            <ngx-material-timepicker-dial-control [timeList]=\"minutes\" [time]=\"minute\" [timeUnit]=\"timeUnit.MINUTE\"\n                                                  [isActive]=\"activeTimeUnit === timeUnit.MINUTE\"\n                                                  [isEditable]=\"isEditable\"\n                                                  [minutesGap]=\"minutesGap\"\n                                                  [disabled]=\"hoursOnly\"\n                                                  (timeUnitChanged)=\"changeTimeUnit($event)\"\n                                                  (timeChanged)=\"changeMinute($event)\"\n                                                  (focused)=\"showHint()\"\n                                                  (unfocused)=\"hideHint()\">\n\n            </ngx-material-timepicker-dial-control>\n        </div>\n        <ngx-material-timepicker-period class=\"timepicker-dial__period\"\n                                        [ngClass]=\"{'timepicker-dial__period--hidden': format === 24}\"\n                                        [selectedPeriod]=\"period\" [activeTimeUnit]=\"activeTimeUnit\"\n                                        [maxTime]=\"maxTime\" [minTime]=\"minTime\" [format]=\"format\"\n                                        [hours]=\"hours\" [minutes]=\"minutes\" [selectedHour]=\"hour\"\n                                        [meridiems]=\"meridiems\"\n                                        (periodChanged)=\"changePeriod($event)\"></ngx-material-timepicker-period>\n    </div>\n    <div *ngIf=\"isEditable || editableHintTmpl\" [ngClass]=\"{'timepicker-dial__hint-container--hidden': !isHintVisible}\">\n        <!--suppress HtmlUnknownAttribute -->\n        <ng-container *ngTemplateOutlet=\"editableHintTmpl ? editableHintTmpl : editableHintDefault\"></ng-container>\n        <ng-template #editableHintDefault>\n            <small class=\"timepicker-dial__hint\"> * use arrows (<span>&#8645;</span>) to change the time</small>\n        </ng-template>\n    </div>\n</div>\n", styles: [".timepicker-dial{text-align:right}.timepicker-dial__container{display:flex;align-items:center;justify-content:flex-end;-webkit-tap-highlight-color:rgba(0,0,0,0)}.timepicker-dial__time{display:flex;align-items:baseline;line-height:normal;font-size:50px;color:#ffffff80;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-dial__time{font-family:var(--primary-font-family);color:var(--dial-inactive-color)}}.timepicker-dial__period{display:block;margin-left:10px}.timepicker-dial__period--hidden{visibility:hidden}.timepicker-dial__hint-container--hidden{visibility:hidden}.timepicker-dial__hint{display:inline-block;font-size:10px;color:#fff}@supports (color: var(--dial-active-color)){.timepicker-dial__hint{color:var(--dial-active-color)}}.timepicker-dial__hint span{font-size:14px}@media (max-device-width: 1023px) and (orientation: landscape){.timepicker-dial__container{flex-direction:column}.timepicker-dial__period{margin-left:0}}\n"], components: [{ type: NgxMaterialTimepickerDialControlComponent, selector: "ngx-material-timepicker-dial-control", inputs: ["timeList", "timeUnit", "time", "isActive", "isEditable", "minutesGap", "disabled"], outputs: ["timeUnitChanged", "timeChanged", "focused", "unfocused"] }, { type: NgxMaterialTimepickerPeriodComponent, selector: "ngx-material-timepicker-period", inputs: ["selectedPeriod", "format", "activeTimeUnit", "hours", "minutes", "minTime", "maxTime", "selectedHour", "meridiems"], outputs: ["periodChanged"] }], directives: [{ type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: i1.NgTemplateOutlet, selector: "[ngTemplateOutlet]", inputs: ["ngTemplateOutletContext", "ngTemplateOutlet"] }], changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerDialComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-dial',
                    templateUrl: 'ngx-material-timepicker-dial.component.html',
                    styleUrls: ['ngx-material-timepicker-dial.component.scss'],
                    changeDetection: ChangeDetectionStrategy.OnPush,
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; }, propDecorators: { editableHintTmpl: [{
                type: Input
            }], hour: [{
                type: Input
            }], minute: [{
                type: Input
            }], format: [{
                type: Input
            }], period: [{
                type: Input
            }], activeTimeUnit: [{
                type: Input
            }], minTime: [{
                type: Input
            }], maxTime: [{
                type: Input
            }], isEditable: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], hoursOnly: [{
                type: Input
            }], periodChanged: [{
                type: Output
            }], timeUnitChanged: [{
                type: Output
            }], hourChanged: [{
                type: Output
            }], minuteChanged: [{
                type: Output
            }] } });

class NgxMaterialTimepickerHoursFace {
    constructor(format) {
        this.hourChange = new EventEmitter();
        this.hourSelected = new EventEmitter();
        this.hoursList = [];
        this.hoursList = TimepickerTimeUtils.getHours(format);
    }
    onTimeSelected(time) {
        this.hourSelected.next(time);
    }
}
NgxMaterialTimepickerHoursFace.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerHoursFace, deps: "invalid", target: i0.ɵɵFactoryTarget.Directive });
NgxMaterialTimepickerHoursFace.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerHoursFace, inputs: { selectedHour: "selectedHour", minTime: "minTime", maxTime: "maxTime", format: "format" }, outputs: { hourChange: "hourChange", hourSelected: "hourSelected" }, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerHoursFace, decorators: [{
            type: Directive
        }], ctorParameters: function () { return [{ type: undefined }]; }, propDecorators: { selectedHour: [{
                type: Input
            }], minTime: [{
                type: Input
            }], maxTime: [{
                type: Input
            }], format: [{
                type: Input
            }], hourChange: [{
                type: Output
            }], hourSelected: [{
                type: Output
            }] } });

class ActiveHourPipe {
    transform(hour, currentHour, isClockFaceDisabled) {
        if (hour == null || isClockFaceDisabled) {
            return false;
        }
        return hour === currentHour;
    }
}
ActiveHourPipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveHourPipe, deps: [], target: i0.ɵɵFactoryTarget.Pipe });
ActiveHourPipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveHourPipe, name: "activeHour" });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveHourPipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'activeHour'
                }]
        }] });

class ActiveMinutePipe {
    transform(minute, currentMinute, gap, isClockFaceDisabled) {
        if (minute == null || isClockFaceDisabled) {
            return false;
        }
        const defaultGap = 5;
        return ((currentMinute === minute) && (minute % (gap || defaultGap) === 0));
    }
}
ActiveMinutePipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveMinutePipe, deps: [], target: i0.ɵɵFactoryTarget.Pipe });
ActiveMinutePipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveMinutePipe, name: "activeMinute" });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: ActiveMinutePipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'activeMinute'
                }]
        }] });

class MinutesFormatterPipe {
    transform(minute, gap = 5) {
        if (!minute) {
            return minute;
        }
        return minute % gap === 0 ? minute : '';
    }
}
MinutesFormatterPipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: MinutesFormatterPipe, deps: [], target: i0.ɵɵFactoryTarget.Pipe });
MinutesFormatterPipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: MinutesFormatterPipe, name: "minutesFormatter" });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: MinutesFormatterPipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'minutesFormatter'
                }]
        }] });

const CLOCK_HAND_STYLES = {
    small: {
        height: '75px',
        top: 'calc(50% - 75px)'
    },
    large: {
        height: '103px',
        top: 'calc(50% - 103px)'
    }
};
class NgxMaterialTimepickerFaceComponent {
    constructor() {
        this.timeUnit = TimeUnit;
        this.innerClockFaceSize = 85;
        this.timeChange = new EventEmitter();
        this.timeSelected = new EventEmitter();
    }
    ngAfterViewInit() {
        this.setClockHandPosition();
        this.addTouchEvents();
    }
    ngOnChanges(changes) {
        const faceTimeChanges = changes['faceTime'];
        const selectedTimeChanges = changes['selectedTime'];
        if ((faceTimeChanges && faceTimeChanges.currentValue)
            && (selectedTimeChanges && selectedTimeChanges.currentValue)) {
            /* Set time according to passed an input value */
            this.selectedTime = this.faceTime.find(time => time.time === this.selectedTime.time);
        }
        if (selectedTimeChanges && selectedTimeChanges.currentValue) {
            this.setClockHandPosition();
        }
        if (faceTimeChanges && faceTimeChanges.currentValue) {
            // To avoid an error ExpressionChangedAfterItHasBeenCheckedError
            setTimeout(() => this.selectAvailableTime());
        }
    }
    trackByTime(_, time) {
        return time.time;
    }
    onMousedown(e) {
        e.preventDefault();
        this.isStarted = true;
    }
    selectTime(e) {
        if (!this.isStarted && (e instanceof MouseEvent && e.type !== 'click')) {
            return;
        }
        const clockFaceCords = this.clockFace.nativeElement.getBoundingClientRect();
        /* Get x0 and y0 of the circle */
        const centerX = clockFaceCords.left + clockFaceCords.width / 2;
        const centerY = clockFaceCords.top + clockFaceCords.height / 2;
        /* Counting the arctangent and convert it to from radian to deg */
        const arctangent = Math.atan(Math.abs(e.clientX - centerX) / Math.abs(e.clientY - centerY)) * 180 / Math.PI;
        /* Get angle according to quadrant */
        const circleAngle = countAngleByCords(centerX, centerY, e.clientX, e.clientY, arctangent);
        /* Check if selected time from the inner clock face (24 hours format only) */
        const isInnerClockChosen = this.format && this.isInnerClockFace(centerX, centerY, e.clientX, e.clientY);
        /* Round angle according to angle step */
        const angleStep = this.unit === TimeUnit.MINUTE ? (6 * (this.minutesGap || 1)) : 30;
        const roundedAngle = roundAngle(circleAngle, angleStep);
        const angle = (roundedAngle || 360) + (isInnerClockChosen ? 360 : 0);
        const selectedTime = this.faceTime.find(val => val.angle === angle);
        if (selectedTime && !selectedTime.disabled) {
            this.timeChange.next(selectedTime);
            /* To let know whether user ended interaction with clock face */
            if (!this.isStarted) {
                this.timeSelected.next(selectedTime.time);
            }
        }
    }
    onMouseup(e) {
        e.preventDefault();
        this.isStarted = false;
    }
    ngOnDestroy() {
        this.removeTouchEvents();
    }
    addTouchEvents() {
        this.touchStartHandler = this.onMousedown.bind(this);
        this.touchEndHandler = this.onMouseup.bind(this);
        this.clockFace.nativeElement.addEventListener('touchstart', this.touchStartHandler);
        this.clockFace.nativeElement.addEventListener('touchend', this.touchEndHandler);
    }
    removeTouchEvents() {
        this.clockFace.nativeElement.removeEventListener('touchstart', this.touchStartHandler);
        this.clockFace.nativeElement.removeEventListener('touchend', this.touchEndHandler);
    }
    setClockHandPosition() {
        if (this.format === 24) {
            if (this.selectedTime.time > 12 || this.selectedTime.time === 0) {
                this.decreaseClockHand();
            }
            else {
                this.increaseClockHand();
            }
        }
        this.clockHand.nativeElement.style.transform = `rotate(${this.selectedTime.angle}deg)`;
    }
    selectAvailableTime() {
        const currentTime = this.faceTime.find(time => this.selectedTime.time === time.time);
        this.isClockFaceDisabled = this.faceTime.every(time => time.disabled);
        if ((currentTime && currentTime.disabled) && !this.isClockFaceDisabled) {
            const availableTime = this.faceTime.find(time => !time.disabled);
            this.timeChange.next(availableTime);
        }
    }
    isInnerClockFace(x0, y0, x, y) {
        /* Detect whether time from the inner clock face or not (24 format only) */
        return Math.sqrt(Math.pow(x - x0, 2) + Math.pow(y - y0, 2)) < this.innerClockFaceSize;
    }
    decreaseClockHand() {
        this.clockHand.nativeElement.style.height = CLOCK_HAND_STYLES.small.height;
        this.clockHand.nativeElement.style.top = CLOCK_HAND_STYLES.small.top;
    }
    increaseClockHand() {
        this.clockHand.nativeElement.style.height = CLOCK_HAND_STYLES.large.height;
        this.clockHand.nativeElement.style.top = CLOCK_HAND_STYLES.large.top;
    }
}
NgxMaterialTimepickerFaceComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerFaceComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerFaceComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerFaceComponent, selector: "ngx-material-timepicker-face", inputs: { faceTime: "faceTime", selectedTime: "selectedTime", unit: "unit", format: "format", minutesGap: "minutesGap" }, outputs: { timeChange: "timeChange", timeSelected: "timeSelected" }, host: { listeners: { "mousedown": "onMousedown($event)", "click": "selectTime($event)", "touchmove": "selectTime($event.changedTouches[0])", "touchend": "selectTime($event.changedTouches[0])", "mousemove": "selectTime($event)", "mouseup": "onMouseup($event)" } }, viewQueries: [{ propertyName: "clockFace", first: true, predicate: ["clockFace"], descendants: true, static: true }, { propertyName: "clockHand", first: true, predicate: ["clockHand"], descendants: true, static: true }], usesOnChanges: true, ngImport: i0, template: "<div class=\"clock-face\" #clockFace>\n    <div *ngIf=\"unit !== timeUnit.MINUTE;else minutesFace\" class=\"clock-face__container\">\n        <div class=\"clock-face__number clock-face__number--outer\"\n             [ngStyle]=\"{'transform': 'rotateZ('+ time.angle +'deg) translateX(-50%)'}\"\n             *ngFor=\"let time of faceTime | slice: 0 : 12; trackBy: trackByTime\">\n\t\t\t<span [ngStyle]=\"{'transform': 'rotateZ(-'+ time.angle +'deg)'}\"\n                  [ngClass]=\"{'active': time.time | activeHour: selectedTime.time : isClockFaceDisabled,\n                   'disabled': time.disabled}\">\n                {{time.time | timeLocalizer: timeUnit.HOUR}}\n            </span>\n        </div>\n        <div class=\"clock-face__inner\" *ngIf=\"faceTime.length > 12\"\n             [style.top]=\"'calc(50% - ' + innerClockFaceSize + 'px)'\">\n            <div class=\"clock-face__number clock-face__number--inner\"\n                 [ngStyle]=\"{'transform': 'rotateZ('+ time.angle +'deg) translateX(-50%)'}\"\n                 [style.height.px]=\"innerClockFaceSize\"\n                 *ngFor=\"let time of faceTime | slice: 12 : 24; trackBy: trackByTime\">\n\t\t\t<span [ngStyle]=\"{'transform': 'rotateZ(-'+ time.angle +'deg)'}\"\n                  [ngClass]=\"{'active': time.time | activeHour: selectedTime?.time : isClockFaceDisabled,\n                   'disabled': time.disabled}\">\n                {{time.time | timeLocalizer: timeUnit.HOUR}}</span>\n            </div>\n        </div>\n    </div>\n\n    <span class=\"clock-face__clock-hand\" [ngClass]=\"{'clock-face__clock-hand_minute': unit === timeUnit.MINUTE}\"\n          #clockHand [hidden]=\"isClockFaceDisabled\"></span>\n</div>\n<ng-template #minutesFace>\n    <div class=\"clock-face__container\">\n        <div class=\"clock-face__number clock-face__number--outer\"\n             [ngStyle]=\"{'transform': 'rotateZ('+ time.angle +'deg) translateX(-50%)'}\"\n             *ngFor=\"let time of faceTime; trackBy: trackByTime\">\n\t<span [ngStyle]=\"{'transform': 'rotateZ(-'+ time.angle +'deg)'}\"\n          [ngClass]=\"{'active': time.time | activeMinute: selectedTime?.time:minutesGap:isClockFaceDisabled,\n           'disabled': time.disabled}\">\n\t{{time.time | minutesFormatter: minutesGap | timeLocalizer: timeUnit.MINUTE}}</span>\n        </div>\n    </div>\n</ng-template>\n", styles: [".clock-face{width:290px;height:290px;border-radius:50%;position:relative;display:flex;justify-content:center;padding:20px;box-sizing:border-box;background-color:#f0f0f0}@supports (background-color: var(--clock-face-background-color)){.clock-face{background-color:var(--clock-face-background-color)}}.clock-face__inner{position:absolute}.clock-face__container{margin-left:-2px}.clock-face__number{position:absolute;transform-origin:0 100%;width:50px;text-align:center;z-index:2}.clock-face__number--outer{height:calc(290px / 2 - 20px)}.clock-face__number--outer>span{font-size:16px;color:#6c6c6c}@supports (color: var(--clock-face-time-inactive-color)){.clock-face__number--outer>span{color:var(--clock-face-time-inactive-color)}}.clock-face__number--inner>span{font-size:14px;color:#929292}@supports (color: var(--clock-face-inner-time-inactive-color)){.clock-face__number--inner>span{color:var(--clock-face-inner-time-inactive-color)}}.clock-face__number>span{-webkit-user-select:none;-moz-user-select:none;user-select:none;width:30px;height:30px;display:flex;justify-content:center;align-items:center;margin:auto;border-radius:50%;font-weight:500;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.clock-face__number>span{font-family:var(--primary-font-family)}}.clock-face__number>span.active{background-color:#00bfff;color:#fff}@supports (background-color: var(--clock-hand-color)){.clock-face__number>span.active{background-color:var(--clock-hand-color);color:var(--clock-face-time-active-color)}}.clock-face__number>span.disabled{color:#c5c5c5}@supports (color: var(--clock-face-time-disabled-color)){.clock-face__number>span.disabled{color:var(--clock-face-time-disabled-color)}}.clock-face__clock-hand{height:103px;width:2px;transform-origin:0 100%;position:absolute;top:calc(50% - 103px);z-index:1;background-color:#00bfff}@supports (background-color: var(--clock-hand-color)){.clock-face__clock-hand{background-color:var(--clock-hand-color)}}.clock-face__clock-hand:after{content:\"\";width:7px;height:7px;border-radius:50%;background-color:inherit;position:absolute;bottom:-3px;left:-3.5px}.clock-face__clock-hand_minute:before{content:\"\";width:7px;height:7px;background-color:#fff;border-radius:50%;position:absolute;top:-8px;left:calc(50% - 8px);box-sizing:content-box;border-width:4px;border-style:solid;border-color:#00bfff}@supports (border-color: var(--clock-hand-color)){.clock-face__clock-hand_minute:before{border-color:var(--clock-hand-color)}}@media (max-device-width: 1023px) and (orientation: landscape){.clock-face{width:225px;height:225px;padding:5px}.clock-face__number--outer{height:calc(225px / 2 - 5px)}.clock-face__clock-hand_minute:before{top:0}}\n"], directives: [{ type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: i1.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { type: i1.NgStyle, selector: "[ngStyle]", inputs: ["ngStyle"] }, { type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }], pipes: { "slice": i1.SlicePipe, "activeHour": ActiveHourPipe, "timeLocalizer": TimeLocalizerPipe, "activeMinute": ActiveMinutePipe, "minutesFormatter": MinutesFormatterPipe }, changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerFaceComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-face',
                    templateUrl: './ngx-material-timepicker-face.component.html',
                    styleUrls: ['./ngx-material-timepicker-face.component.scss'],
                    changeDetection: ChangeDetectionStrategy.OnPush
                }]
        }], propDecorators: { faceTime: [{
                type: Input
            }], selectedTime: [{
                type: Input
            }], unit: [{
                type: Input
            }], format: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], timeChange: [{
                type: Output
            }], timeSelected: [{
                type: Output
            }], clockFace: [{
                type: ViewChild,
                args: ['clockFace', { static: true }]
            }], clockHand: [{
                type: ViewChild,
                args: ['clockHand', { static: true }]
            }], onMousedown: [{
                type: HostListener,
                args: ['mousedown', ['$event']]
            }], selectTime: [{
                type: HostListener,
                args: ['click', ['$event']]
            }, {
                type: HostListener,
                args: ['touchmove', ['$event.changedTouches[0]']]
            }, {
                type: HostListener,
                args: ['touchend', ['$event.changedTouches[0]']]
            }, {
                type: HostListener,
                args: ['mousemove', ['$event']]
            }], onMouseup: [{
                type: HostListener,
                args: ['mouseup', ['$event']]
            }] } });
function roundAngle(angle, step) {
    return Math.round(angle / step) * step;
}
function countAngleByCords(x0, y0, x, y, currentAngle) {
    if (y > y0 && x >= x0) { // II quarter
        return 180 - currentAngle;
    }
    else if (y > y0 && x < x0) { // III quarter
        return 180 + currentAngle;
    }
    else if (y < y0 && x < x0) { // IV quarter
        return 360 - currentAngle;
    }
    else { // I quarter
        return currentAngle;
    }
}

class NgxMaterialTimepicker24HoursFaceComponent extends NgxMaterialTimepickerHoursFace {
    constructor() {
        super(24);
    }
    ngAfterContentInit() {
        this.hoursList = TimepickerTimeUtils.disableHours(this.hoursList, {
            min: this.minTime,
            max: this.maxTime,
            format: this.format
        });
    }
}
NgxMaterialTimepicker24HoursFaceComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepicker24HoursFaceComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepicker24HoursFaceComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepicker24HoursFaceComponent, selector: "ngx-material-timepicker-24-hours-face", usesInheritance: true, ngImport: i0, template: "<ngx-material-timepicker-face [selectedTime]=\"selectedHour\" [faceTime]=\"hoursList\" [format]=\"format\"\n                              (timeChange)=\"hourChange.next($event)\"\n                              (timeSelected)=\"onTimeSelected($event)\"></ngx-material-timepicker-face>\n", components: [{ type: NgxMaterialTimepickerFaceComponent, selector: "ngx-material-timepicker-face", inputs: ["faceTime", "selectedTime", "unit", "format", "minutesGap"], outputs: ["timeChange", "timeSelected"] }], changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepicker24HoursFaceComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-24-hours-face',
                    templateUrl: 'ngx-material-timepicker-24-hours-face.component.html',
                    changeDetection: ChangeDetectionStrategy.OnPush
                }]
        }], ctorParameters: function () { return []; } });

class NgxMaterialTimepicker12HoursFaceComponent extends NgxMaterialTimepickerHoursFace {
    constructor() {
        super(12);
    }
    ngOnChanges(changes) {
        if (changes['period'] && changes['period'].currentValue) {
            this.hoursList = TimepickerTimeUtils.disableHours(this.hoursList, {
                min: this.minTime,
                max: this.maxTime,
                format: this.format,
                period: this.period
            });
        }
    }
}
NgxMaterialTimepicker12HoursFaceComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepicker12HoursFaceComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepicker12HoursFaceComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepicker12HoursFaceComponent, selector: "ngx-material-timepicker-12-hours-face", inputs: { period: "period" }, usesInheritance: true, usesOnChanges: true, ngImport: i0, template: "<ngx-material-timepicker-face [selectedTime]=\"selectedHour\" [faceTime]=\"hoursList\"\n                              (timeChange)=\"hourChange.next($event)\" (timeSelected)=\"onTimeSelected($event)\"></ngx-material-timepicker-face>\n", components: [{ type: NgxMaterialTimepickerFaceComponent, selector: "ngx-material-timepicker-face", inputs: ["faceTime", "selectedTime", "unit", "format", "minutesGap"], outputs: ["timeChange", "timeSelected"] }], changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepicker12HoursFaceComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-12-hours-face',
                    templateUrl: 'ngx-material-timepicker-12-hours-face.component.html',
                    changeDetection: ChangeDetectionStrategy.OnPush
                }]
        }], ctorParameters: function () { return []; }, propDecorators: { period: [{
                type: Input
            }] } });

class NgxMaterialTimepickerMinutesFaceComponent {
    constructor() {
        this.minutesList = [];
        this.timeUnit = TimeUnit;
        this.minuteChange = new EventEmitter();
    }
    ngOnChanges(changes) {
        if (changes['period'] && changes['period'].currentValue) {
            const minutes = TimepickerTimeUtils.getMinutes(this.minutesGap);
            this.minutesList = TimepickerTimeUtils.disableMinutes(minutes, this.selectedHour, {
                min: this.minTime,
                max: this.maxTime,
                format: this.format,
                period: this.period
            });
        }
    }
}
NgxMaterialTimepickerMinutesFaceComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerMinutesFaceComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerMinutesFaceComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerMinutesFaceComponent, selector: "ngx-material-timepicker-minutes-face", inputs: { selectedMinute: "selectedMinute", selectedHour: "selectedHour", period: "period", minTime: "minTime", maxTime: "maxTime", format: "format", minutesGap: "minutesGap" }, outputs: { minuteChange: "minuteChange" }, usesOnChanges: true, ngImport: i0, template: "<ngx-material-timepicker-face [faceTime]=\"minutesList\" [selectedTime]=\"selectedMinute\"\n                              [minutesGap]=\"minutesGap\"\n                              (timeChange)=\"minuteChange.next($event)\" [unit]=\"timeUnit.MINUTE\"></ngx-material-timepicker-face>\n", components: [{ type: NgxMaterialTimepickerFaceComponent, selector: "ngx-material-timepicker-face", inputs: ["faceTime", "selectedTime", "unit", "format", "minutesGap"], outputs: ["timeChange", "timeSelected"] }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerMinutesFaceComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-minutes-face',
                    templateUrl: './ngx-material-timepicker-minutes-face.component.html'
                }]
        }], propDecorators: { selectedMinute: [{
                type: Input
            }], selectedHour: [{
                type: Input
            }], period: [{
                type: Input
            }], minTime: [{
                type: Input
            }], maxTime: [{
                type: Input
            }], format: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], minuteChange: [{
                type: Output
            }] } });

class NgxMaterialTimepickerButtonComponent {
}
NgxMaterialTimepickerButtonComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerButtonComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerButtonComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerButtonComponent, selector: "ngx-material-timepicker-button", ngImport: i0, template: "<button class=\"timepicker-button\" type=\"button\">\n  <span><ng-content></ng-content></span>\n</button>\n", styles: [".timepicker-button{display:inline-block;height:36px;min-width:88px;line-height:36px;border:12px;border-radius:2px;background-color:transparent;text-align:center;transition:all .45s cubic-bezier(.23,1,.32,1);overflow:hidden;-webkit-user-select:none;-moz-user-select:none;user-select:none;position:relative;cursor:pointer;outline:none;color:#00bfff}@supports (color: var(--button-color)){.timepicker-button{color:var(--button-color)}}.timepicker-button:hover,.timepicker-button:focus{background-color:#9993}.timepicker-button>span{font-size:14px;text-transform:uppercase;font-weight:600;padding-left:16px;padding-right:16px;font-family:\"Roboto\",sans-serif}@supports (font-family: var(--primary-font-family)){.timepicker-button>span{font-family:var(--primary-font-family)}}\n"] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerButtonComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-button',
                    templateUrl: './ngx-material-timepicker-button.component.html',
                    styleUrls: ['./ngx-material-timepicker-button.component.scss']
                }]
        }] });

class OverlayDirective {
    constructor(eventService) {
        this.eventService = eventService;
    }
    onClick(e) {
        if (!this.preventClick) {
            this.eventService.dispatchEvent(e);
        }
        e.preventDefault();
    }
}
OverlayDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: OverlayDirective, deps: [{ token: NgxMaterialTimepickerEventService }], target: i0.ɵɵFactoryTarget.Directive });
OverlayDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: OverlayDirective, selector: "[overlay]", inputs: { preventClick: ["overlay", "preventClick"] }, host: { listeners: { "click": "onClick($event)" } }, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: OverlayDirective, decorators: [{
            type: Directive,
            args: [{
                    selector: '[overlay]'
                }]
        }], ctorParameters: function () { return [{ type: NgxMaterialTimepickerEventService }]; }, propDecorators: { preventClick: [{
                type: Input,
                args: ['overlay']
            }], onClick: [{
                type: HostListener,
                args: ['click', ['$event']]
            }] } });

class NgxMaterialTimepickerThemeDirective {
    constructor(elementRef) {
        this.element = elementRef.nativeElement;
    }
    ngAfterViewInit() {
        if (this.theme) {
            this.setTheme(this.theme);
        }
    }
    setTheme(theme) {
        for (const val in theme) {
            if (theme.hasOwnProperty(val)) {
                if (typeof theme[val] === 'string') {
                    for (const prop in theme) {
                        if (theme.hasOwnProperty(prop)) {
                            this.element.style.setProperty(`--${camelCaseToDash(prop)}`, theme[prop]);
                        }
                    }
                    return;
                }
                this.setTheme(theme[val]);
            }
        }
    }
}
NgxMaterialTimepickerThemeDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerThemeDirective, deps: [{ token: i0.ElementRef }], target: i0.ɵɵFactoryTarget.Directive });
NgxMaterialTimepickerThemeDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerThemeDirective, selector: "[ngxMaterialTimepickerTheme]", inputs: { theme: ["ngxMaterialTimepickerTheme", "theme"] }, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerThemeDirective, decorators: [{
            type: Directive,
            args: [{ selector: '[ngxMaterialTimepickerTheme]' }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }]; }, propDecorators: { theme: [{
                type: Input,
                args: ['ngxMaterialTimepickerTheme']
            }] } });
function camelCaseToDash(myStr) {
    return myStr.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
}

var AnimationState;
(function (AnimationState) {
    AnimationState["ENTER"] = "enter";
    AnimationState["LEAVE"] = "leave";
})(AnimationState || (AnimationState = {}));
class NgxMaterialTimepickerContainerComponent {
    constructor(timepickerService, eventService, locale) {
        this.timepickerService = timepickerService;
        this.eventService = eventService;
        this.locale = locale;
        this.timeUnit = TimeUnit;
        this.activeTimeUnit = TimeUnit.HOUR;
        this.unsubscribe = new Subject();
    }
    set defaultTime(time) {
        this._defaultTime = time;
        this.setDefaultTime(time);
    }
    get defaultTime() {
        return this._defaultTime;
    }
    onKeydown(e) {
        this.eventService.dispatchEvent(e);
        e.stopPropagation();
    }
    ngOnInit() {
        this.animationState = !this.disableAnimation && AnimationState.ENTER;
        this.defineTime();
        this.selectedHour = this.timepickerService.selectedHour
            .pipe(shareReplay({ bufferSize: 1, refCount: true }));
        this.selectedMinute = this.timepickerService.selectedMinute
            .pipe(shareReplay({ bufferSize: 1, refCount: true }));
        this.selectedPeriod = this.timepickerService.selectedPeriod
            .pipe(shareReplay({ bufferSize: 1, refCount: true }));
        this.timepickerBaseRef.timeUpdated.pipe(takeUntil(this.unsubscribe))
            .subscribe(this.setDefaultTime.bind(this));
    }
    onHourChange(hour) {
        this.timepickerService.hour = hour;
        this.onTimeChange();
    }
    onHourSelected(hour) {
        if (!this.hoursOnly) {
            this.changeTimeUnit(TimeUnit.MINUTE);
        }
        this.timepickerBaseRef.hourSelected.next(hour);
    }
    onMinuteChange(minute) {
        this.timepickerService.minute = minute;
        this.onTimeChange();
    }
    changePeriod(period) {
        this.timepickerService.period = period;
        this.onTimeChange();
    }
    changeTimeUnit(unit) {
        this.activeTimeUnit = unit;
    }
    setTime() {
        this.timepickerBaseRef.timeSet.next(this.timepickerService.getFullTime(this.format));
        this.close();
    }
    close() {
        if (this.disableAnimation) {
            this.timepickerBaseRef.close();
            return;
        }
        this.animationState = AnimationState.LEAVE;
    }
    animationDone(event) {
        if (event.phaseName === 'done' && event.toState === AnimationState.LEAVE) {
            this.timepickerBaseRef.close();
        }
    }
    ngOnDestroy() {
        this.unsubscribe.next();
        this.unsubscribe.complete();
    }
    setDefaultTime(time) {
        this.timepickerService.setDefaultTimeIfAvailable(time, this.minTime, this.maxTime, this.format, this.minutesGap);
    }
    defineTime() {
        const minTime = this.minTime;
        if (minTime && (!this.time && !this.defaultTime)) {
            const time = TimeAdapter.fromDateTimeToString(minTime, this.format);
            this.setDefaultTime(time);
        }
    }
    onTimeChange() {
        const time = TimeAdapter.toLocaleTimeString(this.timepickerService.getFullTime(this.format), {
            locale: this.locale,
            format: this.format
        });
        this.timepickerBaseRef.timeChanged.emit(time);
    }
}
NgxMaterialTimepickerContainerComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerContainerComponent, deps: [{ token: NgxMaterialTimepickerService }, { token: NgxMaterialTimepickerEventService }, { token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerContainerComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerContainerComponent, selector: "ngx-material-timepicker-container", inputs: { defaultTime: "defaultTime" }, host: { listeners: { "keydown": "onKeydown($event)" } }, providers: [NgxMaterialTimepickerService], ngImport: i0, template: "<div class=\"timepicker-backdrop-overlay\" [overlay]=\"preventOverlayClick\"\n     [ngClass]=\"{'timepicker-backdrop-overlay--transparent': appendToInput}\"></div>\n<div class=\"timepicker-overlay\">\n    <ngx-material-timepicker-content [appendToInput]=\"appendToInput\"\n                                     [inputElement]=\"inputElement\"\n                                     [ngxMaterialTimepickerTheme]=\"theme\">\n        <div class=\"timepicker\"\n             [@timepicker]=\"animationState\"\n             (@timepicker.done)=\"animationDone($event)\"\n             [ngClass]=\"timepickerClass\">\n            <header class=\"timepicker__header\">\n                <ngx-material-timepicker-dial [format]=\"format\" [hour]=\"(selectedHour | async)?.time\"\n                                              [minute]=\"(selectedMinute | async)?.time\"\n                                              [period]=\"selectedPeriod | async\"\n                                              [activeTimeUnit]=\"activeTimeUnit\"\n                                              [minTime]=\"minTime\"\n                                              [maxTime]=\"maxTime\"\n                                              [isEditable]=\"enableKeyboardInput\"\n                                              [editableHintTmpl]=\"editableHintTmpl\"\n                                              [minutesGap]=\"minutesGap\"\n                                              [hoursOnly]=\"hoursOnly\"\n                                              (periodChanged)=\"changePeriod($event)\"\n                                              (timeUnitChanged)=\"changeTimeUnit($event)\"\n                                              (hourChanged)=\"onHourChange($event)\"\n                                              (minuteChanged)=\"onMinuteChange($event)\"\n                ></ngx-material-timepicker-dial>\n            </header>\n            <div class=\"timepicker__main-content\">\n                <div class=\"timepicker__body\" [ngSwitch]=\"activeTimeUnit\">\n                    <div *ngSwitchCase=\"timeUnit.HOUR\">\n                        <ngx-material-timepicker-24-hours-face *ngIf=\"format === 24;else ampmHours\"\n                                                               (hourChange)=\"onHourChange($event)\"\n                                                               [selectedHour]=\"selectedHour | async\"\n                                                               [minTime]=\"minTime\"\n                                                               [maxTime]=\"maxTime\"\n                                                               [format]=\"format\"\n                                                               (hourSelected)=\"onHourSelected($event)\"></ngx-material-timepicker-24-hours-face>\n                        <ng-template #ampmHours>\n                            <ngx-material-timepicker-12-hours-face\n                                (hourChange)=\"onHourChange($event)\"\n                                [selectedHour]=\"selectedHour | async\"\n                                [period]=\"selectedPeriod | async\"\n                                [minTime]=\"minTime\"\n                                [maxTime]=\"maxTime\"\n                                (hourSelected)=\"onHourSelected($event)\"></ngx-material-timepicker-12-hours-face>\n                        </ng-template>\n                    </div>\n                    <ngx-material-timepicker-minutes-face *ngSwitchCase=\"timeUnit.MINUTE\"\n                                                          [selectedMinute]=\"selectedMinute | async\"\n                                                          [selectedHour]=\"(selectedHour | async)?.time\"\n                                                          [minTime]=\"minTime\"\n                                                          [maxTime]=\"maxTime\"\n                                                          [format]=\"format\"\n                                                          [period]=\"selectedPeriod | async\"\n                                                          [minutesGap]=\"minutesGap\"\n                                                          (minuteChange)=\"onMinuteChange($event)\"></ngx-material-timepicker-minutes-face>\n                </div>\n                <div class=\"timepicker__actions\">\n                    <div (click)=\"close()\">\n                        <!--suppress HtmlUnknownAttribute -->\n                        <ng-container\n                            *ngTemplateOutlet=\"cancelBtnTmpl ? cancelBtnTmpl : cancelBtnDefault\"></ng-container>\n                    </div>\n                    <div (click)=\"setTime()\">\n                        <!--suppress HtmlUnknownAttribute -->\n                        <ng-container\n                            *ngTemplateOutlet=\"confirmBtnTmpl ? confirmBtnTmpl : confirmBtnDefault\"></ng-container>\n                    </div>\n                </div>\n            </div>\n        </div>\n    </ngx-material-timepicker-content>\n</div>\n<ng-template #cancelBtnDefault>\n    <ngx-material-timepicker-button>Cancel</ngx-material-timepicker-button>\n</ng-template>\n<ng-template #confirmBtnDefault>\n    <ngx-material-timepicker-button>Ok</ngx-material-timepicker-button>\n</ng-template>\n", styles: [":host{--body-background-color: #fff;--primary-font-family: \"Roboto\", sans-serif;--button-color: deepskyblue;--dial-active-color: #fff;--dial-inactive-color: rgba(255, 255, 255, .5);--dial-background-color: deepskyblue;--dial-editable-active-color: deepskyblue;--dial-editable-background-color: #fff;--clock-face-time-active-color: #fff;--clock-face-time-inactive-color: #6c6c6c;--clock-face-inner-time-inactive-color: #929292;--clock-face-time-disabled-color: #c5c5c5;--clock-face-background-color: #f0f0f0;--clock-hand-color: deepskyblue}.timepicker-backdrop-overlay{position:fixed;top:0;bottom:0;right:0;left:0;background-color:#0000004d;z-index:999;pointer-events:auto}.timepicker-backdrop-overlay--transparent{background-color:transparent}.timepicker-overlay{position:fixed;top:0;left:0;width:100%;height:100%;display:flex;justify-content:center;align-items:center;z-index:999;pointer-events:none}.timepicker{width:300px;border-radius:2px;box-shadow:#00000040 0 14px 45px,#00000038 0 10px 18px;outline:none;position:static;z-index:999;pointer-events:auto}.timepicker__header{padding:15px 30px;background-color:#00bfff}@supports (background-color: var(--dial-background-color)){.timepicker__header{background-color:var(--dial-background-color)}}.timepicker__body{padding:15px 5px;display:flex;justify-content:center;align-items:center;background-color:#fff}@supports (background-color: var(--body-background-color)){.timepicker__body{background-color:var(--body-background-color)}}.timepicker__actions{display:flex;justify-content:flex-end;padding:15px;background-color:#fff}@supports (background-color: var(--body-background-color)){.timepicker__actions{background-color:var(--body-background-color)}}@media (max-device-width: 1023px) and (orientation: landscape){.timepicker{display:flex;width:515px}.timepicker__header{display:flex;align-items:center}.timepicker__main-content{display:flex;flex-direction:column;width:100%}.timepicker__actions{padding:5px;margin-top:-1px}}\n"], components: [{ type: NgxMaterialTimepickerContentComponent, selector: "ngx-material-timepicker-content", inputs: ["appendToInput", "inputElement"] }, { type: NgxMaterialTimepickerDialComponent, selector: "ngx-material-timepicker-dial", inputs: ["editableHintTmpl", "hour", "minute", "format", "period", "activeTimeUnit", "minTime", "maxTime", "isEditable", "minutesGap", "hoursOnly"], outputs: ["periodChanged", "timeUnitChanged", "hourChanged", "minuteChanged"] }, { type: NgxMaterialTimepicker24HoursFaceComponent, selector: "ngx-material-timepicker-24-hours-face" }, { type: NgxMaterialTimepicker12HoursFaceComponent, selector: "ngx-material-timepicker-12-hours-face", inputs: ["period"] }, { type: NgxMaterialTimepickerMinutesFaceComponent, selector: "ngx-material-timepicker-minutes-face", inputs: ["selectedMinute", "selectedHour", "period", "minTime", "maxTime", "format", "minutesGap"], outputs: ["minuteChange"] }, { type: NgxMaterialTimepickerButtonComponent, selector: "ngx-material-timepicker-button" }], directives: [{ type: OverlayDirective, selector: "[overlay]", inputs: ["overlay"] }, { type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: NgxMaterialTimepickerThemeDirective, selector: "[ngxMaterialTimepickerTheme]", inputs: ["ngxMaterialTimepickerTheme"] }, { type: i1.NgSwitch, selector: "[ngSwitch]", inputs: ["ngSwitch"] }, { type: i1.NgSwitchCase, selector: "[ngSwitchCase]", inputs: ["ngSwitchCase"] }, { type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: i1.NgTemplateOutlet, selector: "[ngTemplateOutlet]", inputs: ["ngTemplateOutletContext", "ngTemplateOutlet"] }], pipes: { "async": i1.AsyncPipe }, animations: [
        trigger('timepicker', [
            transition(`* => ${AnimationState.ENTER}`, [
                style({ transform: 'translateY(-30%)' }),
                animate('0.2s ease-out', style({ transform: 'translateY(0)' }))
            ]),
            transition(`${AnimationState.ENTER} => ${AnimationState.LEAVE}`, [
                style({ transform: 'translateY(0)', opacity: 1 }),
                animate('0.2s ease-out', style({ transform: 'translateY(-30%)', opacity: 0 }))
            ])
        ])
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerContainerComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-container',
                    templateUrl: './ngx-material-timepicker-container.component.html',
                    styleUrls: ['./ngx-material-timepicker-container.component.scss'],
                    animations: [
                        trigger('timepicker', [
                            transition(`* => ${AnimationState.ENTER}`, [
                                style({ transform: 'translateY(-30%)' }),
                                animate('0.2s ease-out', style({ transform: 'translateY(0)' }))
                            ]),
                            transition(`${AnimationState.ENTER} => ${AnimationState.LEAVE}`, [
                                style({ transform: 'translateY(0)', opacity: 1 }),
                                animate('0.2s ease-out', style({ transform: 'translateY(-30%)', opacity: 0 }))
                            ])
                        ])
                    ],
                    providers: [NgxMaterialTimepickerService]
                }]
        }], ctorParameters: function () { return [{ type: NgxMaterialTimepickerService }, { type: NgxMaterialTimepickerEventService }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; }, propDecorators: { defaultTime: [{
                type: Input
            }], onKeydown: [{
                type: HostListener,
                args: ['keydown', ['$event']]
            }] } });

class DomService {
    constructor(cfr, appRef, injector, document) {
        this.cfr = cfr;
        this.appRef = appRef;
        this.injector = injector;
        this.document = document;
    }
    appendTimepickerToBody(timepicker, config) {
        this.componentRef = this.cfr.resolveComponentFactory(timepicker).create(this.injector);
        Object.keys(config).forEach(key => this.componentRef.instance[key] = config[key]);
        this.appRef.attachView(this.componentRef.hostView);
        const domElement = this.componentRef.hostView
            .rootNodes[0];
        this.document.body.appendChild(domElement);
    }
    destroyTimepicker() {
        this.componentRef.destroy();
        this.appRef.detachView(this.componentRef.hostView);
    }
}
DomService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: DomService, deps: [{ token: i0.ComponentFactoryResolver }, { token: i0.ApplicationRef }, { token: i0.Injector }, { token: DOCUMENT, optional: true }], target: i0.ɵɵFactoryTarget.Injectable });
DomService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: DomService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: DomService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root'
                }]
        }], ctorParameters: function () { return [{ type: i0.ComponentFactoryResolver }, { type: i0.ApplicationRef }, { type: i0.Injector }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }]; } });

const ESCAPE = 27;
class NgxMaterialTimepickerComponent {
    constructor(eventService, domService) {
        this.eventService = eventService;
        this.domService = domService;
        this.timeUpdated = new Subject();
        this.isEsc = true;
        this.hoursOnly = false;
        this.timeSet = new EventEmitter();
        this.opened = new EventEmitter();
        this.closed = new EventEmitter();
        this.hourSelected = new EventEmitter();
        this.timeChanged = new EventEmitter();
        this.unsubscribe = new Subject();
    }
    /**
     * @deprecated Since version 5.1.1. Will be deleted on version 6.0.0. Use @Input() theme instead
     */
    set ngxMaterialTimepickerTheme(theme) {
        console.warn(`'ngxMaterialTimepickerTheme' is deprecated. Use 'theme' instead`);
        this._ngxMaterialTimepickerTheme = theme;
    }
    set format(value) {
        this._format = value === 24 ? 24 : 12;
    }
    get format() {
        return this.timepickerInput ? this.timepickerInput.format : this._format;
    }
    set minutesGap(gap) {
        if (gap == null) {
            return;
        }
        gap = Math.floor(gap);
        this._minutesGap = gap <= 59 ? gap : 1;
    }
    get minutesGap() {
        return this._minutesGap;
    }
    get minTime() {
        return this.timepickerInput ? this.timepickerInput.min : this.min;
    }
    get maxTime() {
        return this.timepickerInput ? this.timepickerInput.max : this.max;
    }
    get disabled() {
        return this.timepickerInput && this.timepickerInput.disabled;
    }
    get time() {
        return this.timepickerInput && this.timepickerInput.value;
    }
    get inputElement() {
        return this.timepickerInput && this.timepickerInput.element;
    }
    /***
     * Register an input with this timepicker.
     * input - The timepicker input to register with this timepicker
     */
    registerInput(input) {
        if (this.timepickerInput) {
            throw Error('A Timepicker can only be associated with a single input.');
        }
        this.timepickerInput = input;
    }
    open() {
        this.domService.appendTimepickerToBody(NgxMaterialTimepickerContainerComponent, {
            timepickerBaseRef: this,
            time: this.time,
            defaultTime: this.defaultTime,
            maxTime: this.maxTime,
            minTime: this.minTime,
            format: this.format,
            minutesGap: this.minutesGap,
            disableAnimation: this.disableAnimation,
            cancelBtnTmpl: this.cancelBtnTmpl,
            confirmBtnTmpl: this.confirmBtnTmpl,
            editableHintTmpl: this.editableHintTmpl,
            disabled: this.disabled,
            enableKeyboardInput: this.enableKeyboardInput,
            preventOverlayClick: this.preventOverlayClick,
            appendToInput: this.appendToInput,
            hoursOnly: this.hoursOnly,
            theme: this.theme || this._ngxMaterialTimepickerTheme,
            timepickerClass: this.timepickerClass,
            inputElement: this.inputElement
        });
        this.opened.next();
        this.subscribeToEvents();
    }
    close() {
        this.domService.destroyTimepicker();
        this.closed.next();
        this.unsubscribeFromEvents();
    }
    updateTime(time) {
        this.timeUpdated.next(time);
    }
    subscribeToEvents() {
        merge(this.eventService.backdropClick, this.eventService.keydownEvent.pipe(filter(e => e.keyCode === ESCAPE && this.isEsc)))
            .pipe(takeUntil(this.unsubscribe))
            .subscribe(() => this.close());
    }
    unsubscribeFromEvents() {
        this.unsubscribe.next();
        this.unsubscribe.complete();
    }
}
NgxMaterialTimepickerComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerComponent, deps: [{ token: NgxMaterialTimepickerEventService }, { token: DomService }], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerComponent, selector: "ngx-material-timepicker", inputs: { cancelBtnTmpl: "cancelBtnTmpl", editableHintTmpl: "editableHintTmpl", confirmBtnTmpl: "confirmBtnTmpl", isEsc: ["ESC", "isEsc"], enableKeyboardInput: "enableKeyboardInput", preventOverlayClick: "preventOverlayClick", disableAnimation: "disableAnimation", appendToInput: "appendToInput", hoursOnly: "hoursOnly", defaultTime: "defaultTime", timepickerClass: "timepickerClass", theme: "theme", min: "min", max: "max", ngxMaterialTimepickerTheme: "ngxMaterialTimepickerTheme", format: "format", minutesGap: "minutesGap" }, outputs: { timeSet: "timeSet", opened: "opened", closed: "closed", hourSelected: "hourSelected", timeChanged: "timeChanged" }, ngImport: i0, template: '', isInline: true });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker',
                    template: '',
                }]
        }], ctorParameters: function () { return [{ type: NgxMaterialTimepickerEventService }, { type: DomService }]; }, propDecorators: { cancelBtnTmpl: [{
                type: Input
            }], editableHintTmpl: [{
                type: Input
            }], confirmBtnTmpl: [{
                type: Input
            }], isEsc: [{
                type: Input,
                args: ['ESC']
            }], enableKeyboardInput: [{
                type: Input
            }], preventOverlayClick: [{
                type: Input
            }], disableAnimation: [{
                type: Input
            }], appendToInput: [{
                type: Input
            }], hoursOnly: [{
                type: Input
            }], defaultTime: [{
                type: Input
            }], timepickerClass: [{
                type: Input
            }], theme: [{
                type: Input
            }], min: [{
                type: Input
            }], max: [{
                type: Input
            }], ngxMaterialTimepickerTheme: [{
                type: Input
            }], format: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], timeSet: [{
                type: Output
            }], opened: [{
                type: Output
            }], closed: [{
                type: Output
            }], hourSelected: [{
                type: Output
            }], timeChanged: [{
                type: Output
            }] } });

/* To override a default toggle icon */
class NgxMaterialTimepickerToggleIconDirective {
}
NgxMaterialTimepickerToggleIconDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerToggleIconDirective, deps: [], target: i0.ɵɵFactoryTarget.Directive });
NgxMaterialTimepickerToggleIconDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerToggleIconDirective, selector: "[ngxMaterialTimepickerToggleIcon]", ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerToggleIconDirective, decorators: [{
            type: Directive,
            args: [{ selector: '[ngxMaterialTimepickerToggleIcon]' }]
        }] });

class NgxMaterialTimepickerToggleComponent {
    get disabled() {
        return this._disabled === undefined ? this.timepicker.disabled : this._disabled;
    }
    set disabled(value) {
        this._disabled = value;
    }
    open(event) {
        if (this.timepicker) {
            this.timepicker.open();
            event.stopPropagation();
        }
    }
}
NgxMaterialTimepickerToggleComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerToggleComponent, deps: [], target: i0.ɵɵFactoryTarget.Component });
NgxMaterialTimepickerToggleComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxMaterialTimepickerToggleComponent, selector: "ngx-material-timepicker-toggle", inputs: { timepicker: ["for", "timepicker"], disabled: "disabled" }, queries: [{ propertyName: "customIcon", first: true, predicate: NgxMaterialTimepickerToggleIconDirective, descendants: true, static: true }], ngImport: i0, template: "<button class=\"ngx-material-timepicker-toggle\" (click)=\"open($event)\" [disabled]=\"disabled\" type=\"button\">\n    <svg xmlns=\"http://www.w3.org/2000/svg\" viewBox=\"0 0 24 24\" width=\"24px\" height=\"24px\" *ngIf=\"!customIcon\">\n        <path\n            d=\"M 12 2 C 6.4889971 2 2 6.4889971 2 12 C 2 17.511003                   6.4889971 22 12 22 C 17.511003 22 22 17.511003 22 12 C 22 6.4889971 17.511003 2 12 2 z M 12 4 C 16.430123 4 20 7.5698774 20 12 C 20 16.430123 16.430123 20 12 20 C 7.5698774 20 4 16.430123 4 12 C 4 7.5698774 7.5698774 4 12 4 z M 11 6 L 11 12.414062 L 15.292969 16.707031 L 16.707031 15.292969 L 13 11.585938 L 13 6 L 11 6 z\"/>\n    </svg>\n\n    <ng-content select=\"[ngxMaterialTimepickerToggleIcon]\"></ng-content>\n</button>\n", styles: [".ngx-material-timepicker-toggle{display:flex;justify-content:center;align-items:center;padding:4px;background-color:transparent;border-radius:50%;text-align:center;border:none;outline:none;-webkit-user-select:none;-moz-user-select:none;user-select:none;transition:background-color .3s;cursor:pointer}.ngx-material-timepicker-toggle:focus{background-color:#00000012}\n"], directives: [{ type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerToggleComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-material-timepicker-toggle',
                    templateUrl: 'ngx-material-timepicker-toggle.component.html',
                    styleUrls: ['ngx-material-timepicker-toggle.component.scss']
                }]
        }], propDecorators: { timepicker: [{
                type: Input,
                args: ['for']
            }], disabled: [{
                type: Input
            }], customIcon: [{
                type: ContentChild,
                args: [NgxMaterialTimepickerToggleIconDirective, { static: true }]
            }] } });

class TimepickerDirective {
    constructor(elementRef, locale) {
        this.elementRef = elementRef;
        this.locale = locale;
        this._format = 12;
        this._value = '';
        this.timepickerSubscriptions = [];
        this.onTouched = () => {
        };
        this.onChange = () => {
        };
    }
    set format(value) {
        this._format = value === 24 ? 24 : 12;
        const isDynamicallyChanged = value && (this.previousFormat && this.previousFormat !== this._format);
        if (isDynamicallyChanged) {
            this.value = this._value;
            this._timepicker.updateTime(this._value);
        }
        this.previousFormat = this._format;
    }
    get format() {
        return this._format;
    }
    set min(value) {
        console.log(value);
        if (typeof value === 'string') {
            this._min = TimeAdapter.parseTime(value, { locale: this.locale, format: this.format });
            return;
        }
        this._min = value;
    }
    get min() {
        return this._min;
    }
    set max(value) {
        if (typeof value === 'string') {
            this._max = TimeAdapter.parseTime(value, { locale: this.locale, format: this.format });
            return;
        }
        this._max = value;
    }
    get max() {
        return this._max;
    }
    set timepicker(picker) {
        this.registerTimepicker(picker);
    }
    set value(value) {
        if (!value) {
            this._value = '';
            this.updateInputValue();
            return;
        }
        this.setTimeIfAvailable(value);
    }
    get value() {
        if (!this._value) {
            return '';
        }
        return TimeAdapter.toLocaleTimeString(this._value, { format: this.format, locale: this.locale });
    }
    get element() {
        return this.elementRef && this.elementRef.nativeElement;
    }
    set defaultTime(time) {
        this._timepicker.defaultTime = TimeAdapter.formatTime(time, { locale: this.locale, format: this.format });
    }
    updateValue(value) {
        this.value = value;
        this.onChange(value);
    }
    ngOnChanges(changes) {
        var _a;
        const value = (_a = changes === null || changes === void 0 ? void 0 : changes.value) === null || _a === void 0 ? void 0 : _a.currentValue;
        if (value) {
            // Call setTimeIfAvailable after @Input setters
            this.setTimeIfAvailable(value);
            this.defaultTime = value;
        }
    }
    onClick(event) {
        if (!this.disableClick) {
            this._timepicker.open();
            event.stopPropagation();
        }
    }
    writeValue(value) {
        this.value = value;
        if (value) {
            this.defaultTime = value;
        }
    }
    registerOnChange(fn) {
        this.onChange = fn;
    }
    registerOnTouched(fn) {
        this.onTouched = fn;
    }
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
    }
    ngOnDestroy() {
        this.timepickerSubscriptions.forEach(s => s.unsubscribe());
    }
    registerTimepicker(picker) {
        if (picker) {
            this._timepicker = picker;
            this._timepicker.registerInput(this);
            this.timepickerSubscriptions.push(this._timepicker.timeSet.subscribe((time) => {
                this.value = time;
                this.onChange(this.value);
                this.onTouched();
                this.defaultTime = this._value;
            }));
        }
        else {
            throw new Error('NgxMaterialTimepickerComponent is not defined.' +
                ' Please make sure you passed the timepicker to ngxTimepicker directive');
        }
    }
    updateInputValue() {
        this.elementRef.nativeElement.value = this.value;
    }
    setTimeIfAvailable(value) {
        var _a;
        const time = TimeAdapter.formatTime(value, { locale: this.locale, format: this.format });
        const isAvailable = TimeAdapter.isTimeAvailable(time, this._min, this._max, 'minutes', (_a = this._timepicker) === null || _a === void 0 ? void 0 : _a.minutesGap, this._format);
        if (isAvailable) {
            this._value = time;
            this.updateInputValue();
        }
        else {
            this.value = null;
            console.warn('Selected time doesn\'t match min or max value');
        }
    }
}
TimepickerDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimepickerDirective, deps: [{ token: i0.ElementRef }, { token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Directive });
TimepickerDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "12.2.17", type: TimepickerDirective, selector: "[ngxTimepicker]", inputs: { format: "format", min: "min", max: "max", timepicker: ["ngxTimepicker", "timepicker"], value: "value", disabled: "disabled", disableClick: "disableClick" }, host: { listeners: { "change": "updateValue($event.target.value)", "blur": "onTouched()", "click": "onClick($event)" }, properties: { "disabled": "disabled" } }, providers: [
        {
            provide: NG_VALUE_ACCESSOR,
            useExisting: TimepickerDirective,
            multi: true
        }
    ], usesOnChanges: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimepickerDirective, decorators: [{
            type: Directive,
            args: [{
                    selector: '[ngxTimepicker]',
                    providers: [
                        {
                            provide: NG_VALUE_ACCESSOR,
                            useExisting: TimepickerDirective,
                            multi: true
                        }
                    ],
                    host: {
                        '[disabled]': 'disabled',
                        '(change)': 'updateValue($event.target.value)',
                        '(blur)': 'onTouched()',
                    },
                }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; }, propDecorators: { format: [{
                type: Input
            }], min: [{
                type: Input
            }], max: [{
                type: Input
            }], timepicker: [{
                type: Input,
                args: ['ngxTimepicker']
            }], value: [{
                type: Input
            }], disabled: [{
                type: Input
            }], disableClick: [{
                type: Input
            }], onClick: [{
                type: HostListener,
                args: ['click', ['$event']]
            }] } });

class TimeFormatterPipe {
    transform(time, timeUnit) {
        if (time == null || time === '') {
            return time;
        }
        switch (timeUnit) {
            case TimeUnit.HOUR:
                return DateTime.fromObject({ hour: +time }).toFormat('HH');
            case TimeUnit.MINUTE:
                return DateTime.fromObject({ minute: +time }).toFormat('mm');
            default:
                throw new Error('no such time unit');
        }
    }
}
TimeFormatterPipe.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeFormatterPipe, deps: [], target: i0.ɵɵFactoryTarget.Pipe });
TimeFormatterPipe.ɵpipe = i0.ɵɵngDeclarePipe({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeFormatterPipe, name: "timeFormatter" });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: TimeFormatterPipe, decorators: [{
            type: Pipe,
            args: [{
                    name: 'timeFormatter'
                }]
        }] });

class NgxTimepickerTimeControlComponent {
    constructor(timeParser) {
        this.timeParser = timeParser;
        this.timeChanged = new EventEmitter();
    }
    ngOnChanges(changes) {
        if (changes.timeList && this.time != null) {
            if (this.isSelectedTimeDisabled(this.time)) {
                this.setAvailableTime();
            }
        }
    }
    changeTime(event) {
        event.stopPropagation();
        const char = String.fromCharCode(event.keyCode);
        const time = concatTime(String(this.time), char);
        this.changeTimeIfValid(time);
    }
    onKeydown(event) {
        event.stopPropagation();
        if (!isDigit(event)) {
            event.preventDefault();
        }
        switch (event.key) {
            case 'ArrowUp':
                this.increase();
                break;
            case 'ArrowDown':
                this.decrease();
                break;
        }
        if (this.preventTyping && event.key !== 'Tab') {
            event.preventDefault();
        }
    }
    increase() {
        if (!this.disabled) {
            let nextTime = +this.time + (this.minutesGap || 1);
            if (nextTime > this.max) {
                nextTime = this.min;
            }
            if (this.isSelectedTimeDisabled(nextTime)) {
                nextTime = this.getAvailableTime(nextTime, this.getNextAvailableTime.bind(this));
            }
            if (nextTime !== this.time) {
                this.timeChanged.emit(nextTime);
            }
        }
    }
    decrease() {
        if (!this.disabled) {
            let previousTime = +this.time - (this.minutesGap || 1);
            if (previousTime < this.min) {
                previousTime = this.minutesGap ? this.max - (this.minutesGap - 1) : this.max;
            }
            if (this.isSelectedTimeDisabled(previousTime)) {
                previousTime = this.getAvailableTime(previousTime, this.getPrevAvailableTime.bind(this));
            }
            if (previousTime !== this.time) {
                this.timeChanged.emit(previousTime);
            }
        }
    }
    onFocus() {
        this.isFocused = true;
        this.previousTime = this.time;
    }
    onBlur() {
        this.isFocused = false;
        if (this.previousTime !== this.time) {
            this.changeTimeIfValid(+this.time);
        }
    }
    onModelChange(value) {
        this.time = +this.timeParser.transform(value, this.timeUnit);
    }
    changeTimeIfValid(value) {
        if (!isNaN(value)) {
            this.time = value;
            if (this.time > this.max) {
                const timeString = String(value);
                this.time = +timeString[timeString.length - 1];
            }
            if (this.time < this.min) {
                this.time = this.min;
            }
            this.timeChanged.emit(this.time);
        }
    }
    isSelectedTimeDisabled(time) {
        return this.timeList.find((faceTime) => faceTime.time === time).disabled;
    }
    getNextAvailableTime(index) {
        const timeCollection = this.timeList;
        const maxValue = timeCollection.length;
        for (let i = index + 1; i < maxValue; i++) {
            const time = timeCollection[i];
            if (!time.disabled) {
                return time.time;
            }
        }
    }
    getPrevAvailableTime(index) {
        for (let i = index; i >= 0; i--) {
            const time = this.timeList[i];
            if (!time.disabled) {
                return time.time;
            }
        }
    }
    getAvailableTime(currentTime, fn) {
        const currentTimeIndex = this.timeList.findIndex(time => time.time === currentTime);
        const availableTime = fn(currentTimeIndex);
        return availableTime != null ? availableTime : this.time;
    }
    setAvailableTime() {
        const availableTime = this.timeList.find(t => !t.disabled);
        if (availableTime != null) {
            this.time = availableTime.time;
            this.timeChanged.emit(this.time);
        }
    }
}
NgxTimepickerTimeControlComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerTimeControlComponent, deps: [{ token: TimeParserPipe }], target: i0.ɵɵFactoryTarget.Component });
NgxTimepickerTimeControlComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxTimepickerTimeControlComponent, selector: "ngx-timepicker-time-control", inputs: { time: "time", min: "min", max: "max", placeholder: "placeholder", timeUnit: "timeUnit", disabled: "disabled", timeList: "timeList", preventTyping: "preventTyping", minutesGap: "minutesGap" }, outputs: { timeChanged: "timeChanged" }, providers: [TimeParserPipe], usesOnChanges: true, ngImport: i0, template: "<div class=\"ngx-timepicker-control\" [ngClass]=\"{'ngx-timepicker-control--active': isFocused}\">\n    <!--suppress HtmlFormInputWithoutLabel -->\n    <input class=\"ngx-timepicker-control__input\"\n           maxlength=\"2\"\n           [ngModel]=\"time | timeParser: timeUnit | timeLocalizer: timeUnit : true\"\n           (ngModelChange)=\"onModelChange($event)\"\n           [placeholder]=\"placeholder\"\n           [disabled]=\"disabled\"\n           (keydown)=\"onKeydown($event)\"\n           (keypress)=\"changeTime($event)\"\n           (focus)=\"onFocus()\"\n           (blur)=\"onBlur()\">\n    <div class=\"ngx-timepicker-control__arrows\">\n            <span class=\"ngx-timepicker-control__arrow\" role=\"button\" (click)=\"increase()\">\n                &#9650;\n            </span>\n        <span class=\"ngx-timepicker-control__arrow\" role=\"button\" (click)=\"decrease()\">\n                &#9660;\n            </span>\n    </div>\n</div>\n", styles: [".ngx-timepicker-control{position:relative;display:flex;width:60px;height:30px;padding:0 5px;box-sizing:border-box}.ngx-timepicker-control--active:after{content:\"\";position:absolute;bottom:-2px;left:0;width:100%;height:1px;background-color:#00bfff}.ngx-timepicker-control__input{width:100%;height:100%;padding:0 5px 0 0;border:0;font-size:1rem;color:inherit;outline:none;text-align:center}.ngx-timepicker-control__input:disabled{background-color:transparent}.ngx-timepicker-control__arrows{position:absolute;right:2px;top:0;display:flex;flex-direction:column}.ngx-timepicker-control__arrow{font-size:11px;color:#0006;cursor:pointer;transition:color .2s;-webkit-user-select:none;-moz-user-select:none;user-select:none}.ngx-timepicker-control__arrow:hover{color:#000000e6}\n"], directives: [{ type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i4.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { type: i4.MaxLengthValidator, selector: "[maxlength][formControlName],[maxlength][formControl],[maxlength][ngModel]", inputs: ["maxlength"] }, { type: i4.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { type: i4.NgModel, selector: "[ngModel]:not([formControlName]):not([formControl])", inputs: ["name", "disabled", "ngModel", "ngModelOptions"], outputs: ["ngModelChange"], exportAs: ["ngModel"] }], pipes: { "timeLocalizer": TimeLocalizerPipe, "timeParser": TimeParserPipe }, changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerTimeControlComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-timepicker-time-control',
                    templateUrl: './ngx-timepicker-time-control.component.html',
                    styleUrls: ['./ngx-timepicker-time-control.component.scss'],
                    changeDetection: ChangeDetectionStrategy.OnPush,
                    providers: [TimeParserPipe],
                }]
        }], ctorParameters: function () { return [{ type: TimeParserPipe }]; }, propDecorators: { time: [{
                type: Input
            }], min: [{
                type: Input
            }], max: [{
                type: Input
            }], placeholder: [{
                type: Input
            }], timeUnit: [{
                type: Input
            }], disabled: [{
                type: Input
            }], timeList: [{
                type: Input
            }], preventTyping: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], timeChanged: [{
                type: Output
            }] } });
function concatTime(currentTime, nextTime) {
    const isNumber = /\d/.test(nextTime);
    if (isNumber) {
        const time = currentTime + nextTime;
        return +time;
    }
}

class NgxTimepickerPeriodSelectorComponent {
    constructor(locale) {
        this.locale = locale;
        this.periodSelected = new EventEmitter();
        this.period = TimePeriod;
        this.meridiems = Info.meridiems({ locale: this.locale });
    }
    set selectedPeriod(period) {
        if (period) {
            const periods = [TimePeriod.AM, TimePeriod.PM];
            this.localizedPeriod = this.meridiems[periods.indexOf(period)];
        }
    }
    open() {
        if (!this.disabled) {
            this.isOpened = true;
        }
    }
    select(period) {
        this.periodSelected.next(period);
        this.isOpened = false;
    }
    backdropClick() {
        this.isOpened = false;
    }
}
NgxTimepickerPeriodSelectorComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerPeriodSelectorComponent, deps: [{ token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Component });
NgxTimepickerPeriodSelectorComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxTimepickerPeriodSelectorComponent, selector: "ngx-timepicker-period-selector", inputs: { isOpened: "isOpened", disabled: "disabled", selectedPeriod: "selectedPeriod" }, outputs: { periodSelected: "periodSelected" }, ngImport: i0, template: "<div class=\"period\">\n    <div class=\"period-control\">\n        <button class=\"period-control__button period__btn--default\"\n                [ngClass]=\"{'period-control__button--disabled': disabled}\"\n                type=\"button\"\n                (click)=\"open()\">\n            <span>{{localizedPeriod}}</span>\n            <span class=\"period-control__arrow\">&#9660;</span>\n        </button>\n    </div>\n    <ul class=\"period-selector\" @scaleInOut *ngIf=\"isOpened\" [timepickerAutofocus]=\"true\">\n        <li>\n            <button class=\"period-selector__button period__btn--default\"\n                    type=\"button\"\n                    (click)=\"select(period.AM)\"\n                    [ngClass]=\"{'period-selector__button--active': localizedPeriod === meridiems[0]}\">{{meridiems[0]}}</button>\n        </li>\n        <li>\n            <button class=\"period-selector__button period__btn--default\"\n                    type=\"button\"\n                    (click)=\"select(period.PM)\"\n                    [ngClass]=\"{'period-selector__button--active': localizedPeriod === meridiems[1]}\">{{meridiems[1]}}</button>\n        </li>\n    </ul>\n</div>\n<div class=\"overlay\" (click)=\"backdropClick()\" *ngIf=\"isOpened\"></div>\n", styles: [".period{position:relative}.period__btn--default{padding:0;border:none;background-color:transparent;cursor:pointer;text-align:left;-webkit-user-select:none;-moz-user-select:none;user-select:none;-webkit-tap-highlight-color:transparent;outline:none}.period-control{position:relative}.period-control__button{position:relative;width:60px;font-size:1rem;color:inherit;text-align:center}.period-control__button:not(.period-control__button--disabled):focus:after{content:\"\";position:absolute;bottom:-8px;left:0;width:100%;height:1px;background-color:#00bfff}.period-control__arrow{margin-left:10px;font-size:12px;color:#0006}.period-selector{position:absolute;top:calc(50% - 50px);right:calc(-50% + -50px);max-width:135px;width:150px;padding:6px 0;margin:0;list-style:none;background-color:#f5f5f5;box-shadow:0 1px 3px #0003,0 1px 1px #00000024,0 2px 1px -1px #0000001f;z-index:201}.period-selector__button{width:100%;height:48px;padding:0 16px;line-height:48px}.period-selector__button--active{color:#00bfff}.period-selector__button:focus{background-color:#eee}.overlay{position:fixed;width:100%;height:100%;top:0;left:0;background-color:transparent;z-index:200}\n"], directives: [{ type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: AutofocusDirective, selector: "[timepickerAutofocus]", inputs: ["timepickerAutofocus"] }], animations: [
        trigger('scaleInOut', [
            transition(':enter', [
                style({ transform: 'scale(0)', opacity: 0 }),
                animate(200, style({ transform: 'scale(1)', opacity: 1 }))
            ]),
            transition(':leave', [
                animate(200, style({ transform: 'scale(0)', opacity: 0 }))
            ])
        ])
    ], changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerPeriodSelectorComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-timepicker-period-selector',
                    templateUrl: 'ngx-timepicker-period-selector.component.html',
                    styleUrls: ['./ngx-timepicker-period-selector.component.scss'],
                    changeDetection: ChangeDetectionStrategy.OnPush,
                    animations: [
                        trigger('scaleInOut', [
                            transition(':enter', [
                                style({ transform: 'scale(0)', opacity: 0 }),
                                animate(200, style({ transform: 'scale(1)', opacity: 1 }))
                            ]),
                            transition(':leave', [
                                animate(200, style({ transform: 'scale(0)', opacity: 0 }))
                            ])
                        ])
                    ]
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; }, propDecorators: { isOpened: [{
                type: Input
            }], disabled: [{
                type: Input
            }], selectedPeriod: [{
                type: Input
            }], periodSelected: [{
                type: Output
            }] } });

class NgxTimepickerFieldComponent {
    constructor(timepickerService, locale) {
        this.timepickerService = timepickerService;
        this.locale = locale;
        this.minHour = 1;
        this.maxHour = 12;
        this.timeUnit = TimeUnit;
        this.buttonAlign = 'right';
        this.timeChanged = new EventEmitter();
        this._format = 12;
        this.unsubscribe$ = new Subject();
        this.isFirstTimeChange = true;
        this.onChange = () => {
        };
    }
    set format(value) {
        this._format = value === 24 ? 24 : 12;
        this.minHour = this._format === 12 ? 1 : 0;
        this.maxHour = this._format === 12 ? 12 : 23;
        this.hoursList = TimepickerTimeUtils.getHours(this._format);
        const isDynamicallyChanged = value && (this.previousFormat && this.previousFormat !== this._format);
        if (isDynamicallyChanged) {
            this.updateTime(this.timepickerTime);
        }
        this.previousFormat = this._format;
    }
    get format() {
        return this._format;
    }
    set min(value) {
        if (typeof value === 'string') {
            this._min = TimeAdapter.parseTime(value, { locale: this.locale, format: this.format });
            return;
        }
        this._min = value;
    }
    get min() {
        return this._min;
    }
    set max(value) {
        if (typeof value === 'string') {
            this._max = TimeAdapter.parseTime(value, { locale: this.locale, format: this.format });
            return;
        }
        this._max = value;
    }
    get max() {
        return this._max;
    }
    set defaultTime(val) {
        this._defaultTime = val;
        this.isDefaultTime = !!val;
    }
    get defaultTime() {
        return this._defaultTime;
    }
    set minutesGap(gap) {
        if (gap == null) {
            return;
        }
        gap = Math.floor(gap);
        this._minutesGap = gap <= 59 ? gap : 1;
    }
    get minutesGap() {
        return this._minutesGap;
    }
    ngOnInit() {
        this.initTime(this.defaultTime);
        this.hoursList = TimepickerTimeUtils.getHours(this._format);
        this.minutesList = TimepickerTimeUtils.getMinutes();
        this.isTimeRangeSet = !!(this.min || this.max);
        this.hour$ = this.timepickerService.selectedHour.pipe(tap((clockTime) => this.selectedHour = clockTime.time), map(this.changeDefaultTimeValue.bind(this)), tap(() => this.isTimeRangeSet && this.updateAvailableMinutes()));
        this.minute$ = this.timepickerService.selectedMinute.pipe(map(this.changeDefaultTimeValue.bind(this)), tap(() => this.isFirstTimeChange = false));
        if (this.format === 12) {
            this.timepickerService.selectedPeriod.pipe(distinctUntilChanged(), tap((period) => this.period = period), tap(period => this.isChangePeriodDisabled = this.isPeriodDisabled(period)), takeUntil(this.unsubscribe$)).subscribe(() => this.isTimeRangeSet && this.updateAvailableTime());
        }
        else if (this.isTimeRangeSet) {
            this.updateAvailableTime();
        }
    }
    writeValue(val) {
        if (val) {
            this.initTime(val);
        }
        else {
            this.resetTime();
        }
    }
    registerOnTouched(fn) {
    }
    registerOnChange(fn) {
        this.onChange = fn;
    }
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
    }
    changeHour(hour) {
        this.timepickerService.hour = this.hoursList.find(h => h.time === hour);
        this.changeTime();
    }
    changeMinute(minute) {
        this.timepickerService.minute = this.minutesList.find(m => m.time === minute);
        this.changeTime();
    }
    changePeriod(period) {
        this.timepickerService.period = period;
        this.changeTime();
    }
    onTimeSet(time) {
        this.updateTime(time);
        this.emitLocalTimeChange(time);
    }
    ngOnDestroy() {
        this.unsubscribe$.next();
        this.unsubscribe$.complete();
    }
    changeTime() {
        const time = this.timepickerService.getFullTime(this.format);
        this.timepickerTime = time;
        this.emitLocalTimeChange(time);
    }
    resetTime() {
        this.timepickerService.hour = { angle: 0, time: null };
        this.timepickerService.minute = { angle: 0, time: null };
    }
    emitLocalTimeChange(time) {
        const localTime = TimeAdapter.toLocaleTimeString(time, { format: this.format, locale: this.locale });
        this.onChange(localTime);
        this.timeChanged.emit(localTime);
    }
    changeDefaultTimeValue(clockFaceTime) {
        if (!this.isDefaultTime && this.isFirstTimeChange) {
            return Object.assign(Object.assign({}, clockFaceTime), { time: null });
        }
        return clockFaceTime;
    }
    updateAvailableHours() {
        this.hoursList = TimepickerTimeUtils.disableHours(this.hoursList, {
            min: this.min,
            max: this.max,
            format: this.format,
            period: this.period
        });
    }
    updateAvailableMinutes() {
        this.minutesList = TimepickerTimeUtils.disableMinutes(this.minutesList, this.selectedHour, {
            min: this.min,
            max: this.max,
            format: this.format,
            period: this.period
        });
    }
    updateAvailableTime() {
        this.updateAvailableHours();
        if (this.selectedHour) {
            this.updateAvailableMinutes();
        }
    }
    updateTime(time) {
        if (time) {
            const formattedTime = TimeAdapter.formatTime(time, { locale: this.locale, format: this.format });
            this.timepickerService.setDefaultTimeIfAvailable(formattedTime, this.min, this.max, this.format);
            this.timepickerTime = formattedTime;
        }
    }
    initTime(time) {
        const isDefaultTimeAvailable = TimeAdapter
            .isTimeAvailable(time, this.min, this.max, 'minutes', null, this.format);
        if (!isDefaultTimeAvailable) {
            if (this.min) {
                this.updateTime(TimeAdapter.fromDateTimeToString(this.min, this.format));
                return;
            }
            if (this.max) {
                this.updateTime(TimeAdapter.fromDateTimeToString(this.max, this.format));
                return;
            }
        }
        this.updateTime(time);
    }
    isPeriodDisabled(period) {
        return TimepickerTimeUtils.disableHours(TimepickerTimeUtils.getHours(12), {
            min: this.min,
            max: this.max,
            format: 12,
            period: period === TimePeriod.AM ? TimePeriod.PM : TimePeriod.AM
        }).every(time => time.disabled);
    }
}
NgxTimepickerFieldComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerFieldComponent, deps: [{ token: NgxMaterialTimepickerService }, { token: TIME_LOCALE }], target: i0.ɵɵFactoryTarget.Component });
NgxTimepickerFieldComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "12.2.17", type: NgxTimepickerFieldComponent, selector: "ngx-timepicker-field", inputs: { disabled: "disabled", toggleIcon: "toggleIcon", buttonAlign: "buttonAlign", clockTheme: "clockTheme", controlOnly: "controlOnly", cancelBtnTmpl: "cancelBtnTmpl", confirmBtnTmpl: "confirmBtnTmpl", format: "format", min: "min", max: "max", defaultTime: "defaultTime", minutesGap: "minutesGap" }, outputs: { timeChanged: "timeChanged" }, providers: [
        NgxMaterialTimepickerService,
        {
            provide: NG_VALUE_ACCESSOR,
            useExisting: NgxTimepickerFieldComponent,
            multi: true
        }
    ], ngImport: i0, template: "<div class=\"ngx-timepicker\" [ngClass]=\"{'ngx-timepicker--disabled': disabled}\">\n    <ngx-timepicker-time-control\n        class=\"ngx-timepicker__control--first\"\n        [placeholder]=\"'HH'\"\n        [time]=\"(hour$ | async)?.time\"\n        [min]=\"minHour\"\n        [max]=\"maxHour\"\n        [timeUnit]=\"timeUnit.HOUR\"\n        [disabled]=\"disabled\"\n        [timeList]=\"hoursList\"\n        [preventTyping]=\"isTimeRangeSet\"\n        (timeChanged)=\"changeHour($event)\"></ngx-timepicker-time-control>\n    <span class=\"ngx-timepicker__time-colon ngx-timepicker__control--second\">:</span>\n    <ngx-timepicker-time-control\n        class=\"ngx-timepicker__control--third\"\n        [placeholder]=\"'MM'\"\n        [time]=\"(minute$ | async)?.time\"\n        [min]=\"0\"\n        [max]=\"59\"\n        [timeUnit]=\"timeUnit.MINUTE\"\n        [disabled]=\"disabled\"\n        [timeList]=\"minutesList\"\n        [preventTyping]=\"isTimeRangeSet\"\n        [minutesGap]=\"minutesGap\"\n        (timeChanged)=\"changeMinute($event)\"></ngx-timepicker-time-control>\n    <ngx-timepicker-period-selector\n        class=\"ngx-timepicker__control--forth\"\n        [selectedPeriod]=\"period\"\n        [disabled]=\"disabled || isChangePeriodDisabled\"\n        (periodSelected)=\"changePeriod($event)\"\n        *ngIf=\"format !== 24\"></ngx-timepicker-period-selector>\n    <ngx-material-timepicker-toggle\n        class=\"ngx-timepicker__toggle\"\n        *ngIf=\"!controlOnly\"\n        [ngClass]=\"{'ngx-timepicker__toggle--left': buttonAlign === 'left'}\"\n        [for]=\"timepicker\"\n        [disabled]=\"disabled\">\n        <span ngxMaterialTimepickerToggleIcon>\n            <!--suppress HtmlUnknownAttribute -->\n            <ng-container *ngTemplateOutlet=\"toggleIcon || defaultIcon\"></ng-container>\n        </span>\n    </ngx-material-timepicker-toggle>\n</div>\n<ngx-material-timepicker\n    [min]=\"min\"\n    [max]=\"max\"\n    [theme]=\"clockTheme\"\n    [defaultTime]=\"timepickerTime\"\n    [format]=\"format\"\n    [cancelBtnTmpl]=\"cancelBtnTmpl\"\n    [confirmBtnTmpl]=\"confirmBtnTmpl\"\n    [minutesGap]=\"minutesGap\"\n    (timeSet)=\"onTimeSet($event)\" #timepicker></ngx-material-timepicker>\n\n<ng-template #defaultIcon>\n    <svg xmlns=\"http://www.w3.org/2000/svg\" viewBox=\"0 0 24 24\" width=\"24px\" height=\"24px\">\n        <!--suppress CheckEmptyScriptTag -->\n        <path\n            d=\"M 12 2 C 6.4889971 2 2 6.4889971 2 12 C 2 17.511003                   6.4889971 22 12 22 C 17.511003 22 22 17.511003 22 12 C 22 6.4889971 17.511003 2 12 2 z M 12 4 C 16.430123 4 20 7.5698774 20 12 C 20 16.430123 16.430123 20 12 20 C 7.5698774 20 4 16.430123 4 12 C 4 7.5698774 7.5698774 4 12 4 z M 11 6 L 11 12.414062 L 15.292969 16.707031 L 16.707031 15.292969 L 13 11.585938 L 13 6 L 11 6 z\"/>\n    </svg>\n</ng-template>\n", styles: [".ngx-timepicker{display:flex;align-items:center;height:100%;border-bottom:1px solid rgba(0,0,0,.12)}.ngx-timepicker--disabled{background:rgba(0,0,0,.07);pointer-events:none}.ngx-timepicker__time-colon{margin-left:10px}.ngx-timepicker__control--first{order:1}.ngx-timepicker__control--second{order:2}.ngx-timepicker__control--third{order:3}.ngx-timepicker__control--forth{order:4}.ngx-timepicker__toggle{order:4}.ngx-timepicker__toggle--left{order:0}\n"], components: [{ type: NgxTimepickerTimeControlComponent, selector: "ngx-timepicker-time-control", inputs: ["time", "min", "max", "placeholder", "timeUnit", "disabled", "timeList", "preventTyping", "minutesGap"], outputs: ["timeChanged"] }, { type: NgxTimepickerPeriodSelectorComponent, selector: "ngx-timepicker-period-selector", inputs: ["isOpened", "disabled", "selectedPeriod"], outputs: ["periodSelected"] }, { type: NgxMaterialTimepickerToggleComponent, selector: "ngx-material-timepicker-toggle", inputs: ["for", "disabled"] }, { type: NgxMaterialTimepickerComponent, selector: "ngx-material-timepicker", inputs: ["cancelBtnTmpl", "editableHintTmpl", "confirmBtnTmpl", "ESC", "enableKeyboardInput", "preventOverlayClick", "disableAnimation", "appendToInput", "hoursOnly", "defaultTime", "timepickerClass", "theme", "min", "max", "ngxMaterialTimepickerTheme", "format", "minutesGap"], outputs: ["timeSet", "opened", "closed", "hourSelected", "timeChanged"] }], directives: [{ type: i1.NgClass, selector: "[ngClass]", inputs: ["class", "ngClass"] }, { type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { type: NgxMaterialTimepickerToggleIconDirective, selector: "[ngxMaterialTimepickerToggleIcon]" }, { type: i1.NgTemplateOutlet, selector: "[ngTemplateOutlet]", inputs: ["ngTemplateOutletContext", "ngTemplateOutlet"] }], pipes: { "async": i1.AsyncPipe }, changeDetection: i0.ChangeDetectionStrategy.OnPush });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxTimepickerFieldComponent, decorators: [{
            type: Component,
            args: [{
                    selector: 'ngx-timepicker-field',
                    templateUrl: './ngx-timepicker-field.component.html',
                    styleUrls: ['./ngx-timepicker-field.component.scss'],
                    providers: [
                        NgxMaterialTimepickerService,
                        {
                            provide: NG_VALUE_ACCESSOR,
                            useExisting: NgxTimepickerFieldComponent,
                            multi: true
                        }
                    ],
                    changeDetection: ChangeDetectionStrategy.OnPush
                }]
        }], ctorParameters: function () { return [{ type: NgxMaterialTimepickerService }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TIME_LOCALE]
                }] }]; }, propDecorators: { disabled: [{
                type: Input
            }], toggleIcon: [{
                type: Input
            }], buttonAlign: [{
                type: Input
            }], clockTheme: [{
                type: Input
            }], controlOnly: [{
                type: Input
            }], cancelBtnTmpl: [{
                type: Input
            }], confirmBtnTmpl: [{
                type: Input
            }], format: [{
                type: Input
            }], min: [{
                type: Input
            }], max: [{
                type: Input
            }], defaultTime: [{
                type: Input
            }], minutesGap: [{
                type: Input
            }], timeChanged: [{
                type: Output
            }] } });

class NgxMaterialTimepickerModule {
    // tslint:disable-next-line:max-line-length
    static setOpts(locale, numberingSystem = TimeAdapter.DEFAULT_NUMBERING_SYSTEM) {
        return {
            ngModule: NgxMaterialTimepickerModule,
            providers: [
                { provide: TIME_LOCALE, useValue: locale },
                { provide: NUMBERING_SYSTEM, useValue: numberingSystem }
            ]
        };
    }
}
NgxMaterialTimepickerModule.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerModule, deps: [], target: i0.ɵɵFactoryTarget.NgModule });
NgxMaterialTimepickerModule.ɵmod = i0.ɵɵngDeclareNgModule({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerModule, declarations: [NgxMaterialTimepickerComponent,
        NgxMaterialTimepicker24HoursFaceComponent,
        NgxMaterialTimepicker12HoursFaceComponent,
        NgxMaterialTimepickerMinutesFaceComponent,
        NgxMaterialTimepickerFaceComponent,
        NgxMaterialTimepickerToggleComponent,
        NgxMaterialTimepickerButtonComponent,
        NgxMaterialTimepickerDialComponent,
        NgxMaterialTimepickerDialControlComponent,
        NgxMaterialTimepickerPeriodComponent,
        TimeFormatterPipe,
        TimepickerDirective,
        OverlayDirective,
        NgxMaterialTimepickerToggleIconDirective,
        AutofocusDirective,
        MinutesFormatterPipe,
        NgxMaterialTimepickerThemeDirective,
        NgxTimepickerFieldComponent,
        NgxTimepickerTimeControlComponent,
        NgxTimepickerPeriodSelectorComponent,
        TimeLocalizerPipe,
        TimeParserPipe,
        ActiveHourPipe,
        ActiveMinutePipe,
        NgxMaterialTimepickerContainerComponent,
        NgxMaterialTimepickerContentComponent,
        AppendToInputDirective], imports: [CommonModule,
        FormsModule,
        ReactiveFormsModule], exports: [NgxMaterialTimepickerComponent,
        NgxMaterialTimepickerToggleComponent,
        NgxTimepickerFieldComponent,
        TimepickerDirective,
        NgxMaterialTimepickerToggleIconDirective,
        NgxMaterialTimepickerThemeDirective] });
NgxMaterialTimepickerModule.ɵinj = i0.ɵɵngDeclareInjector({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerModule, imports: [[
            CommonModule,
            FormsModule,
            ReactiveFormsModule,
        ]] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerModule, decorators: [{
            type: NgModule,
            args: [{
                    imports: [
                        CommonModule,
                        FormsModule,
                        ReactiveFormsModule,
                    ],
                    exports: [
                        NgxMaterialTimepickerComponent,
                        NgxMaterialTimepickerToggleComponent,
                        NgxTimepickerFieldComponent,
                        TimepickerDirective,
                        NgxMaterialTimepickerToggleIconDirective,
                        NgxMaterialTimepickerThemeDirective,
                    ],
                    declarations: [
                        NgxMaterialTimepickerComponent,
                        NgxMaterialTimepicker24HoursFaceComponent,
                        NgxMaterialTimepicker12HoursFaceComponent,
                        NgxMaterialTimepickerMinutesFaceComponent,
                        NgxMaterialTimepickerFaceComponent,
                        NgxMaterialTimepickerToggleComponent,
                        NgxMaterialTimepickerButtonComponent,
                        NgxMaterialTimepickerDialComponent,
                        NgxMaterialTimepickerDialControlComponent,
                        NgxMaterialTimepickerPeriodComponent,
                        TimeFormatterPipe,
                        TimepickerDirective,
                        OverlayDirective,
                        NgxMaterialTimepickerToggleIconDirective,
                        AutofocusDirective,
                        MinutesFormatterPipe,
                        NgxMaterialTimepickerThemeDirective,
                        NgxTimepickerFieldComponent,
                        NgxTimepickerTimeControlComponent,
                        NgxTimepickerPeriodSelectorComponent,
                        TimeLocalizerPipe,
                        TimeParserPipe,
                        ActiveHourPipe,
                        ActiveMinutePipe,
                        NgxMaterialTimepickerContainerComponent,
                        NgxMaterialTimepickerContentComponent,
                        AppendToInputDirective
                    ],
                }]
        }] });

/**
 * Generated bundle index. Do not edit.
 */

export { NUMBERING_SYSTEM, NgxMaterialTimepickerComponent, NgxMaterialTimepickerModule, NgxMaterialTimepickerThemeDirective, NgxMaterialTimepickerToggleComponent, NgxMaterialTimepickerToggleIconDirective, NgxTimepickerFieldComponent, TIME_LOCALE, TimepickerDirective };
//# sourceMappingURL=ngx-material-timepicker.js.map
