import { DateTime } from 'luxon';
import { TimeFormat } from '../models/time-format.enum';
import { TimePeriod } from '../models/time-period.enum';
import { isBetween, isSameOrAfter, isSameOrBefore } from '../utils/timepicker.utils';
// @dynamic
export class TimeAdapter {
    static parseTime(time, opts) {
        const { numberingSystem, locale } = TimeAdapter.getLocaleOptionsByTime(time, opts);
        const isPeriodExist = time.split(' ').length === 2;
        const timeMask = isPeriodExist ? TimeFormat.TWELVE_SHORT : TimeFormat.TWENTY_FOUR_SHORT;
        return DateTime.fromFormat(time, timeMask, { numberingSystem, locale });
    }
    static formatTime(time, opts) {
        if (!time) {
            return 'Invalid Time';
        }
        const { format } = opts;
        const parsedTime = TimeAdapter.parseTime(time, opts).setLocale(TimeAdapter.DEFAULT_LOCALE);
        if (!parsedTime.isValid) {
            return null;
        }
        if (format !== 24) {
            return parsedTime.toLocaleString(Object.assign(Object.assign({}, DateTime.TIME_SIMPLE), { hour12: format !== 24, numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM })).replace(/\u200E/g, '').replace(/\u202F/g, ' ');
        }
        return parsedTime.toISOTime({
            includeOffset: false,
            suppressMilliseconds: true,
            suppressSeconds: true
        }).replace(/\u200E/g, '').replace(/\u202F/g, ' ');
    }
    static toLocaleTimeString(time, opts = {}) {
        const { format = TimeAdapter.DEFAULT_FORMAT, locale = TimeAdapter.DEFAULT_LOCALE } = opts;
        const hourCycle = format === 24 ? 'h23' : 'h12';
        const timeFormat = Object.assign(Object.assign({}, DateTime.TIME_SIMPLE), { hourCycle });
        const timeMask = (format === 24) ? TimeFormat.TWENTY_FOUR_SHORT : TimeFormat.TWELVE_SHORT;
        const localOpts = Object.assign({ locale: opts.locale, numberingSystem: opts.numberingSystem }, timeFormat);
        return DateTime.fromFormat(time, timeMask).setLocale(locale).toLocaleString(localOpts).replace(/\u202F/g, ' ');
    }
    static isTimeAvailable(time, min, max, granularity, minutesGap, format) {
        if (!time) {
            return;
        }
        const convertedTime = this.parseTime(time, { format });
        const minutes = convertedTime.minute;
        if (minutesGap && minutes === minutes && minutes % minutesGap !== 0) {
            throw new Error(`Your minutes - ${minutes} doesn\'t match your minutesGap - ${minutesGap}`);
        }
        const isAfter = (min && !max)
            && isSameOrAfter(convertedTime, min, granularity);
        const isBefore = (max && !min)
            && isSameOrBefore(convertedTime, max, granularity);
        const between = (min && max)
            && isBetween(convertedTime, min, max, granularity);
        const isAvailable = !min && !max;
        return isAfter || isBefore || between || isAvailable;
    }
    /***
     *  Format hour according to time format (12 or 24)
     */
    static formatHour(currentHour, format, period) {
        if (format === 24) {
            return currentHour;
        }
        const hour = period === TimePeriod.AM ? currentHour : currentHour + 12;
        if (period === TimePeriod.AM && hour === 12) {
            return 0;
        }
        else if (period === TimePeriod.PM && hour === 24) {
            return 12;
        }
        return hour;
    }
    static fromDateTimeToString(time, format) {
        const timeFormat = format === 24 ? TimeFormat.TWENTY_FOUR : TimeFormat.TWELVE;
        return time.reconfigure({
            numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM,
            locale: TimeAdapter.DEFAULT_LOCALE
        }).toFormat(timeFormat).replace(/\u202F/g, ' ');
    }
    static getLocaleOptionsByTime(time, opts) {
        const localeConfig = { numberingSystem: opts.numberingSystem, locale: opts.locale };
        const defaultConfig = { numberingSystem: TimeAdapter.DEFAULT_NUMBERING_SYSTEM, locale: TimeAdapter.DEFAULT_LOCALE };
        return isNaN(parseInt(time, 10)) ? localeConfig : defaultConfig;
    }
}
TimeAdapter.DEFAULT_FORMAT = 12;
TimeAdapter.DEFAULT_LOCALE = 'en-US';
TimeAdapter.DEFAULT_NUMBERING_SYSTEM = 'latn';
//# sourceMappingURL=data:application/json;base64,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