import { Injectable } from '@angular/core';
import { BehaviorSubject } from 'rxjs';
import { TimePeriod } from '../models/time-period.enum';
import { TimeAdapter } from './time-adapter';
import { DateTime } from 'luxon';
import * as i0 from "@angular/core";
const DEFAULT_HOUR = {
    time: 12,
    angle: 360
};
const DEFAULT_MINUTE = {
    time: 0,
    angle: 360
};
export class NgxMaterialTimepickerService {
    constructor() {
        this.hourSubject = new BehaviorSubject(DEFAULT_HOUR);
        this.minuteSubject = new BehaviorSubject(DEFAULT_MINUTE);
        this.periodSubject = new BehaviorSubject(TimePeriod.AM);
    }
    set hour(hour) {
        this.hourSubject.next(hour);
    }
    get selectedHour() {
        return this.hourSubject.asObservable();
    }
    set minute(minute) {
        this.minuteSubject.next(minute);
    }
    get selectedMinute() {
        return this.minuteSubject.asObservable();
    }
    set period(period) {
        const isPeriodValid = (period === TimePeriod.AM) || (period === TimePeriod.PM);
        if (isPeriodValid) {
            this.periodSubject.next(period);
        }
    }
    get selectedPeriod() {
        return this.periodSubject.asObservable();
    }
    setDefaultTimeIfAvailable(time, min, max, format, minutesGap) {
        /* Workaround to double error message*/
        try {
            if (TimeAdapter.isTimeAvailable(time, min, max, 'minutes', minutesGap)) {
                this.setDefaultTime(time, format);
            }
        }
        catch (e) {
            console.error(e);
        }
    }
    getFullTime(format) {
        const selectedHour = this.hourSubject.getValue().time;
        const selectedMinute = this.minuteSubject.getValue().time;
        const hour = selectedHour != null ? selectedHour : DEFAULT_HOUR.time;
        const minute = selectedMinute != null ? selectedMinute : DEFAULT_MINUTE.time;
        const period = format === 12 ? this.periodSubject.getValue() : '';
        const time = `${hour}:${minute} ${period}`.trim();
        return TimeAdapter.formatTime(time, { format });
    }
    setDefaultTime(time, format) {
        const defaultTime = TimeAdapter.parseTime(time, { format }).toJSDate();
        if (DateTime.fromJSDate(defaultTime).isValid) {
            const period = time.substr(time.length - 2).toUpperCase();
            const hour = defaultTime.getHours();
            this.hour = Object.assign(Object.assign({}, DEFAULT_HOUR), { time: formatHourByPeriod(hour, period) });
            this.minute = Object.assign(Object.assign({}, DEFAULT_MINUTE), { time: defaultTime.getMinutes() });
            this.period = period;
        }
        else {
            this.resetTime();
        }
    }
    resetTime() {
        this.hour = Object.assign({}, DEFAULT_HOUR);
        this.minute = Object.assign({}, DEFAULT_MINUTE);
        this.period = TimePeriod.AM;
    }
}
NgxMaterialTimepickerService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, deps: [], target: i0.ɵɵFactoryTarget.Injectable });
NgxMaterialTimepickerService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.17", ngImport: i0, type: NgxMaterialTimepickerService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root'
                }]
        }] });
/***
 *  Format hour in 24hours format to meridian (AM or PM) format
 */
function formatHourByPeriod(hour, period) {
    switch (period) {
        case TimePeriod.AM:
            return hour === 0 ? 12 : hour;
        case TimePeriod.PM:
            return hour === 12 ? 12 : hour - 12;
        default:
            return hour;
    }
}
//# sourceMappingURL=data:application/json;base64,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