"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addImportsToModuleDeclaration = exports.addImportsToModuleFile = exports.applyChanges = exports.getModuleFile = void 0;
const schematics_1 = require("@angular-devkit/schematics");
const typescript_1 = require("typescript");
const schematics_consts_1 = require("../schematics.consts");
const array_1 = require("../utils/array");
const ast_utils_1 = require("../utils/ast-utils");
const change_1 = require("../utils/change");
const find_module_1 = require("../utils/find-module");
const projects_1 = require("../utils/projects");
const translations_1 = require("../utils/translations");
const transloco_1 = require("../utils/transloco");
const schema_1 = require("./schema");
const path = require("path");
function jsonTranslationFileCreator(source, lang) {
    return source.create(`${lang}.json`, `{}
`);
}
function createTranslateFiles(langs, creator) {
    const treeSource = new schematics_1.EmptyTree();
    langs.forEach((lang) => {
        creator(treeSource, lang);
    });
    return treeSource;
}
function getModuleFile(host, options) {
    const modulePath = options.module;
    if (!host.exists(modulePath)) {
        throw new schematics_1.SchematicsException(`File ${modulePath} does not exist.`);
    }
    const text = host.read(modulePath);
    if (text === null) {
        throw new schematics_1.SchematicsException(`File ${modulePath} does not exist.`);
    }
    const sourceText = text.toString('utf-8');
    return (0, typescript_1.createSourceFile)(modulePath, sourceText, typescript_1.ScriptTarget.Latest, true);
}
exports.getModuleFile = getModuleFile;
function applyChanges(host, path, changes) {
    const recorder = host.beginUpdate(path);
    for (const change of changes) {
        if (change instanceof change_1.InsertChange) {
            recorder.insertLeft(change.pos, change.toAdd);
        }
    }
    host.commitUpdate(recorder);
    return host;
}
exports.applyChanges = applyChanges;
function addImportsToModuleFile(options, imports, file = schematics_consts_1.LIB_NAME) {
    return (host) => {
        const module = getModuleFile(host, options);
        const importChanges = (0, ast_utils_1.insertImport)(module, options.module, imports.join(', '), file);
        return applyChanges(host, options.module, [
            importChanges,
        ]);
    };
}
exports.addImportsToModuleFile = addImportsToModuleFile;
function addImportsToModuleDeclaration(options, imports) {
    return (host) => {
        const module = getModuleFile(host, options);
        const importChanges = imports.map((imp) => (0, ast_utils_1.addImportToModule)(module, options.module, imp, schematics_consts_1.LIB_NAME)[0]);
        return applyChanges(host, options.module, importChanges);
    };
}
exports.addImportsToModuleDeclaration = addImportsToModuleDeclaration;
function createTranslocoModule({ isLib, ssr, langs, modulePath, sourceRoot, host }) {
    const envPath = path.relative(modulePath, `${sourceRoot}/environments/environment`).split(path.sep).join('/');
    const envFileExists = host.exists(`${sourceRoot}/environments/environment.ts`);
    let prodMode = envFileExists ? 'environment.production' : '!isDevMode()';
    if (isLib) {
        prodMode = 'false';
    }
    return (0, schematics_1.apply)((0, schematics_1.url)(`./files/transloco-module`), [
        (0, schematics_1.template)({
            ts: 'ts',
            stringifyList: array_1.stringifyList,
            isLib,
            langs,
            importEnv: ssr || envFileExists,
            envPath,
            loaderPrefix: ssr ? '${environment.baseUrl}' : '',
            prodMode,
        }),
        (0, schematics_1.move)('/', modulePath),
    ]);
}
function updateEnvironmentBaseUrl(host, sourceRoot, defaultValue) {
    const template = `$1{
  baseUrl: '${defaultValue}',`;
    (0, projects_1.setEnvironments)(host, sourceRoot, (env) => env.indexOf('baseUrl') === -1
        ? env.replace(/(environment.*=*)\{/, template)
        : env);
}
function default_1(options) {
    return (host, context) => {
        const langs = options.langs.split(',').map((l) => l.trim());
        const project = (0, projects_1.getProject)(host, options.project);
        const sourceRoot = (project && project.sourceRoot) || 'src';
        const isLib = project.projectType === 'library';
        const assetsPath = `${sourceRoot}/${options.path}`;
        const translationCreator = jsonTranslationFileCreator;
        const translateFiles = (0, schematics_1.apply)((0, schematics_1.source)(createTranslateFiles(langs, translationCreator)), [(0, schematics_1.move)('/', assetsPath)]);
        options.module = (0, find_module_1.findRootModule)(host, options.module, sourceRoot);
        const modulePath = options.module.substring(0, options.module.lastIndexOf('/') + 1);
        if (options.ssr) {
            updateEnvironmentBaseUrl(host, sourceRoot, 'http://localhost:4200');
        }
        (0, transloco_1.createConfig)(host, langs, assetsPath);
        return (0, schematics_1.chain)([
            options.loader === schema_1.Loaders.Http
                ? (0, schematics_1.chain)([
                    addImportsToModuleFile(options, ['HttpClientModule'], '@angular/common/http'),
                    addImportsToModuleDeclaration(options, ['HttpClientModule']),
                ])
                : (0, schematics_1.noop)(),
            (0, translations_1.checkIfTranslationFilesExist)(assetsPath, langs, '.json', true)
                ? (0, schematics_1.noop)()
                : (0, schematics_1.mergeWith)(translateFiles),
            (0, schematics_1.mergeWith)(createTranslocoModule({
                sourceRoot,
                isLib,
                ssr: options.ssr,
                langs,
                modulePath,
                host
            })),
            addImportsToModuleFile(options, ['TranslocoRootModule'], './transloco-root.module'),
            addImportsToModuleDeclaration(options, ['TranslocoRootModule']),
        ])(host, context);
    };
}
exports.default = default_1;
//# sourceMappingURL=index.js.map