import { Inject, Injectable, InjectionToken, Injector, Optional, } from '@angular/core';
import { getValue, isDefined, isObject, isString, setValue } from './helpers';
import { defaultConfig, TRANSLOCO_CONFIG, } from './transloco.config';
import * as i0 from "@angular/core";
export const TRANSLOCO_TRANSPILER = new InjectionToken('TRANSLOCO_TRANSPILER');
export class DefaultTranspiler {
    constructor(userConfig) {
        this.interpolationMatcher = resolveMatcher(userConfig);
    }
    transpile(value, params = {}, translation, key) {
        if (isString(value)) {
            return value.replace(this.interpolationMatcher, (_, match) => {
                match = match.trim();
                if (isDefined(params[match])) {
                    return params[match];
                }
                return isDefined(translation[match])
                    ? this.transpile(translation[match], params, translation, key)
                    : '';
            });
        }
        else if (params) {
            if (isObject(value)) {
                value = this.handleObject(value, params, translation, key);
            }
            else if (Array.isArray(value)) {
                value = this.handleArray(value, params, translation, key);
            }
        }
        return value;
    }
    /**
     *
     * @example
     *
     * const en = {
     *  a: {
     *    b: {
     *      c: "Hello {{ value }}"
     *    }
     *  }
     * }
     *
     * const params =  {
     *  "b.c": { value: "Transloco "}
     * }
     *
     * service.selectTranslate('a', params);
     *
     * // the first param will be the result of `en.a`.
     * // the second param will be `params`.
     * parser.transpile(value, params, {});
     *
     *
     */
    handleObject(value, params = {}, translation, key) {
        let result = value;
        Object.keys(params).forEach((p) => {
            // get the value of "b.c" inside "a" => "Hello {{ value }}"
            const v = getValue(result, p);
            // get the params of "b.c" => { value: "Transloco" }
            const getParams = getValue(params, p);
            // transpile the value => "Hello Transloco"
            const transpiled = this.transpile(v, getParams, translation, key);
            // set "b.c" to `transpiled`
            result = setValue(result, p, transpiled);
        });
        return result;
    }
    handleArray(value, params = {}, translation, key) {
        return value.map((v) => this.transpile(v, params, translation, key));
    }
}
DefaultTranspiler.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: DefaultTranspiler, deps: [{ token: TRANSLOCO_CONFIG, optional: true }], target: i0.ɵɵFactoryTarget.Injectable });
DefaultTranspiler.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: DefaultTranspiler });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: DefaultTranspiler, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TRANSLOCO_CONFIG]
                }] }]; } });
function resolveMatcher(userConfig) {
    const [start, end] = userConfig && userConfig.interpolation
        ? userConfig.interpolation
        : defaultConfig.interpolation;
    return new RegExp(`${start}(.*?)${end}`, 'g');
}
export function getFunctionArgs(argsString) {
    const splitted = argsString ? argsString.split(',') : [];
    const args = [];
    for (let i = 0; i < splitted.length; i++) {
        let value = splitted[i].trim();
        while (value[value.length - 1] === '\\') {
            i++;
            value = value.replace('\\', ',') + splitted[i];
        }
        args.push(value);
    }
    return args;
}
export class FunctionalTranspiler extends DefaultTranspiler {
    constructor(injector) {
        super();
        this.injector = injector;
    }
    transpile(value, params = {}, translation, key) {
        let transpiled = value;
        if (isString(value)) {
            transpiled = value.replace(/\[\[\s*(\w+)\((.*)\)\s*]]/g, (match, functionName, args) => {
                try {
                    const func = this.injector.get(functionName);
                    return func.transpile(...getFunctionArgs(args));
                }
                catch (e) {
                    let message = `There is an error in: '${value}'. 
                          Check that the you used the right syntax in your translation and that the implementation of ${functionName} is correct.`;
                    if (e.message.includes('NullInjectorError')) {
                        message = `You are using the '${functionName}' function in your translation but no provider was found!`;
                    }
                    throw new Error(message);
                }
            });
        }
        return super.transpile(transpiled, params, translation, key);
    }
}
FunctionalTranspiler.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: FunctionalTranspiler, deps: [{ token: i0.Injector }], target: i0.ɵɵFactoryTarget.Injectable });
FunctionalTranspiler.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: FunctionalTranspiler });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: FunctionalTranspiler, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i0.Injector }]; } });
//# sourceMappingURL=data:application/json;base64,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