import { Inject, Injectable, Optional } from '@angular/core';
import { BehaviorSubject, combineLatest, EMPTY, forkJoin, from, of, Subject, } from 'rxjs';
import { catchError, map, retry, shareReplay, switchMap, tap, } from 'rxjs/operators';
import { DefaultLoader, TRANSLOCO_LOADER, } from './transloco.loader';
import { TRANSLOCO_TRANSPILER, } from './transloco.transpiler';
import { flatten, isEmpty, isNil, isScopeObject, isString, size, toCamelCase, unflatten, } from './helpers';
import { defaultConfig, TRANSLOCO_CONFIG, } from './transloco.config';
import { TRANSLOCO_MISSING_HANDLER, } from './transloco-missing-handler';
import { TRANSLOCO_INTERCEPTOR, } from './transloco.interceptor';
import { TRANSLOCO_FALLBACK_STRATEGY, } from './transloco-fallback-strategy';
import { mergeConfig } from './merge-config';
import { getEventPayload, getLangFromScope, getScopeFromLang, resolveInlineLoader, } from './shared';
import { getFallbacksLoaders } from './get-fallbacks-loaders';
import { resolveLoader } from './resolve-loader';
import * as i0 from "@angular/core";
let service;
export function translate(key, params = {}, lang) {
    return service.translate(key, params, lang);
}
export function translateObject(key, params = {}, lang) {
    return service.translateObject(key, params, lang);
}
export class TranslocoService {
    constructor(loader, parser, missingHandler, interceptor, userConfig, fallbackStrategy) {
        this.loader = loader;
        this.parser = parser;
        this.missingHandler = missingHandler;
        this.interceptor = interceptor;
        this.userConfig = userConfig;
        this.fallbackStrategy = fallbackStrategy;
        this.subscription = null;
        this.translations = new Map();
        this.cache = new Map();
        this.defaultLang = '';
        this.availableLangs = [];
        this.isResolvedMissingOnce = false;
        this.failedLangs = new Set();
        this.events = new Subject();
        this.events$ = this.events.asObservable();
        if (!this.loader) {
            this.loader = new DefaultLoader(this.translations);
        }
        service = this;
        this.mergedConfig = mergeConfig(defaultConfig, this.userConfig);
        this.setAvailableLangs(this.mergedConfig.availableLangs || []);
        this.setFallbackLangForMissingTranslation(this.mergedConfig);
        this.setDefaultLang(this.mergedConfig.defaultLang);
        this.lang = new BehaviorSubject(this.getDefaultLang());
        // Don't use distinctUntilChanged as we need the ability to update
        // the value when using setTranslation or setTranslationKeys
        this.langChanges$ = this.lang.asObservable();
        /**
         * When we have a failure, we want to define the next language that succeeded as the active
         */
        this.subscription = this.events$.subscribe((e) => {
            if (e.type === 'translationLoadSuccess' && e.wasFailure) {
                this.setActiveLang(e.payload.langName);
            }
        });
    }
    get config() {
        return this.mergedConfig;
    }
    getDefaultLang() {
        return this.defaultLang;
    }
    setDefaultLang(lang) {
        this.defaultLang = lang;
    }
    getActiveLang() {
        return this.lang.getValue();
    }
    setActiveLang(lang) {
        this.parser.onLangChanged?.(lang);
        this.lang.next(lang);
        this.events.next({
            type: 'langChanged',
            payload: getEventPayload(lang),
        });
        return this;
    }
    setAvailableLangs(langs) {
        this.availableLangs = langs;
    }
    /**
     * Gets the available languages.
     *
     * @returns
     * An array of the available languages. Can be either a `string[]` or a `{ id: string; label: string }[]`
     * depending on how the available languages are set in your module.
     */
    getAvailableLangs() {
        return this.availableLangs;
    }
    load(path, options = {}) {
        const cached = this.cache.get(path);
        if (cached) {
            return cached;
        }
        let loadTranslation;
        const isScope = this._isLangScoped(path);
        let scope;
        if (isScope) {
            scope = getScopeFromLang(path);
        }
        const loadersOptions = {
            path,
            mainLoader: this.loader,
            inlineLoader: options.inlineLoader,
            data: isScope ? { scope: scope } : undefined,
        };
        if (this.useFallbackTranslation(path)) {
            // if the path is scope the fallback should be `scope/fallbackLang`;
            const fallback = isScope
                ? `${scope}/${this.firstFallbackLang}`
                : this.firstFallbackLang;
            const loaders = getFallbacksLoaders({
                ...loadersOptions,
                fallbackPath: fallback,
            });
            loadTranslation = forkJoin(loaders);
        }
        else {
            const loader = resolveLoader(loadersOptions);
            loadTranslation = from(loader);
        }
        const load$ = loadTranslation.pipe(retry(this.config.failedRetries), tap((translation) => {
            if (Array.isArray(translation)) {
                translation.forEach((t) => {
                    this.handleSuccess(t.lang, t.translation);
                    // Save the fallback in cache so we'll not create a redundant request
                    if (t.lang !== path) {
                        this.cache.set(t.lang, of({}));
                    }
                });
                return;
            }
            this.handleSuccess(path, translation);
        }), catchError((error) => {
            if (!this.mergedConfig.prodMode) {
                console.error(`Error while trying to load "${path}"`, error);
            }
            return this.handleFailure(path, options);
        }), shareReplay(1));
        this.cache.set(path, load$);
        return load$;
    }
    /**
     * Gets the instant translated value of a key
     *
     * @example
     *
     * translate<string>('hello')
     * translate('hello', { value: 'value' })
     * translate<string[]>(['hello', 'key'])
     * translate('hello', { }, 'en')
     * translate('scope.someKey', { }, 'en')
     */
    translate(key, params = {}, lang = this.getActiveLang()) {
        if (!key)
            return key;
        const { scope, resolveLang } = this.resolveLangAndScope(lang);
        if (Array.isArray(key)) {
            return key.map((k) => this.translate(scope ? `${scope}.${k}` : k, params, resolveLang));
        }
        key = scope ? `${scope}.${key}` : key;
        const translation = this.getTranslation(resolveLang);
        const value = translation[key];
        if (!value) {
            return this._handleMissingKey(key, value, params);
        }
        return this.parser.transpile(value, params, translation, key);
    }
    /**
     * Gets the translated value of a key as observable
     *
     * @example
     *
     * selectTranslate<string>('hello').subscribe(value => ...)
     * selectTranslate<string>('hello', {}, 'es').subscribe(value => ...)
     * selectTranslate<string>('hello', {}, 'todos').subscribe(value => ...)
     * selectTranslate<string>('hello', {}, { scope: 'todos' }).subscribe(value => ...)
     *
     */
    selectTranslate(key, params, lang, _isObject = false) {
        let inlineLoader;
        const load = (lang, options) => this.load(lang, options).pipe(map(() => _isObject
            ? this.translateObject(key, params, lang)
            : this.translate(key, params, lang)));
        if (isNil(lang)) {
            return this.langChanges$.pipe(switchMap((lang) => load(lang)));
        }
        if (isScopeObject(lang)) {
            // it's a scope object.
            const providerScope = lang;
            lang = providerScope.scope;
            inlineLoader = resolveInlineLoader(providerScope, providerScope.scope);
        }
        lang = lang;
        if (this.isLang(lang) || this.isScopeWithLang(lang)) {
            return load(lang);
        }
        // it's a scope
        const scope = lang;
        return this.langChanges$.pipe(switchMap((lang) => load(`${scope}/${lang}`, { inlineLoader })));
    }
    /**
     * Whether the scope with lang
     *
     * @example
     *
     * todos/en => true
     * todos => false
     */
    isScopeWithLang(lang) {
        return this.isLang(getLangFromScope(lang));
    }
    translateObject(key, params = {}, lang = this.getActiveLang()) {
        if (isString(key) || Array.isArray(key)) {
            const { resolveLang, scope } = this.resolveLangAndScope(lang);
            if (Array.isArray(key)) {
                return key.map((k) => this.translateObject(scope ? `${scope}.${k}` : k, params, resolveLang));
            }
            const translation = this.getTranslation(resolveLang);
            key = scope ? `${scope}.${key}` : key;
            const value = unflatten(this.getObjectByKey(translation, key));
            /* If an empty object was returned we want to try and translate the key as a string and not an object */
            return isEmpty(value)
                ? this.translate(key, params, lang)
                : this.parser.transpile(value, params, translation, key);
        }
        const translations = [];
        for (const [_key, _params] of this.getEntries(key)) {
            translations.push(this.translateObject(_key, _params, lang));
        }
        return translations;
    }
    selectTranslateObject(key, params, lang) {
        if (isString(key) || Array.isArray(key)) {
            return this.selectTranslate(key, params, lang, true);
        }
        const [[firstKey, firstParams], ...rest] = this.getEntries(key);
        /* In order to avoid subscribing multiple times to the load language event by calling selectTranslateObject for each pair,
         * we listen to when the first key has been translated (the language is loaded) and translate the rest synchronously */
        return this.selectTranslateObject(firstKey, firstParams, lang).pipe(map((value) => {
            const translations = [value];
            for (const [_key, _params] of rest) {
                translations.push(this.translateObject(_key, _params, lang));
            }
            return translations;
        }));
    }
    getTranslation(langOrScope) {
        if (langOrScope) {
            if (this.isLang(langOrScope)) {
                return this.translations.get(langOrScope) || {};
            }
            else {
                // This is a scope, build the scope value from the translation object
                const { scope, resolveLang } = this.resolveLangAndScope(langOrScope);
                const translation = this.translations.get(resolveLang) || {};
                return this.getObjectByKey(translation, scope);
            }
        }
        return this.translations;
    }
    /**
     * Gets an object of translations for a given language
     *
     * @example
     *
     * selectTranslation().subscribe() - will return the current lang translation
     * selectTranslation('es').subscribe()
     * selectTranslation('admin-page').subscribe() - will return the current lang scope translation
     * selectTranslation('admin-page/es').subscribe()
     */
    selectTranslation(lang) {
        let language$ = this.langChanges$;
        if (lang) {
            const scopeLangSpecified = getLangFromScope(lang) !== lang;
            if (this.isLang(lang) || scopeLangSpecified) {
                language$ = of(lang);
            }
            else {
                language$ = this.langChanges$.pipe(map((currentLang) => `${lang}/${currentLang}`));
            }
        }
        return language$.pipe(switchMap((language) => this.load(language).pipe(map(() => this.getTranslation(language)))));
    }
    /**
     * Sets or merge a given translation object to current lang
     *
     * @example
     *
     * setTranslation({ ... })
     * setTranslation({ ... }, 'en')
     * setTranslation({ ... }, 'es', { merge: false } )
     * setTranslation({ ... }, 'todos/en', { merge: false } )
     */
    setTranslation(translation, lang = this.getActiveLang(), options = {}) {
        const defaults = { merge: true, emitChange: true };
        const mergedOptions = { ...defaults, ...options };
        const scope = getScopeFromLang(lang);
        /**
         * If this isn't a scope we use the whole translation as is
         * otherwise we need to flat the scope and use it
         */
        let flattenScopeOrTranslation = translation;
        // Merged the scoped language into the active language
        if (scope) {
            const key = this.getMappedScope(scope);
            flattenScopeOrTranslation = flatten({ [key]: translation });
        }
        const currentLang = scope ? getLangFromScope(lang) : lang;
        const mergedTranslation = {
            ...(mergedOptions.merge && this.getTranslation(currentLang)),
            ...flattenScopeOrTranslation,
        };
        const flattenTranslation = this.mergedConfig.flatten.aot
            ? mergedTranslation
            : flatten(mergedTranslation);
        const withHook = this.interceptor.preSaveTranslation(flattenTranslation, currentLang);
        this.translations.set(currentLang, withHook);
        mergedOptions.emitChange && this.setActiveLang(this.getActiveLang());
    }
    /**
     * Sets translation key with given value
     *
     * @example
     *
     * setTranslationKey('key', 'value')
     * setTranslationKey('key.nested', 'value')
     * setTranslationKey('key.nested', 'value', 'en')
     * setTranslationKey('key.nested', 'value', 'en', { emitChange: false } )
     */
    setTranslationKey(key, value, 
    // Todo: Add the lang to the options in v3
    lang = this.getActiveLang(), options = {}) {
        const withHook = this.interceptor.preSaveTranslationKey(key, value, lang);
        const newValue = {
            [key]: withHook,
        };
        this.setTranslation(newValue, lang, { ...options, merge: true });
    }
    /**
     * Sets the fallback lang for the currently active language
     * @param fallbackLang
     */
    setFallbackLangForMissingTranslation({ fallbackLang, }) {
        const lang = Array.isArray(fallbackLang) ? fallbackLang[0] : fallbackLang;
        if (fallbackLang && this.useFallbackTranslation(lang)) {
            this.firstFallbackLang = lang;
        }
    }
    /**
     * @internal
     */
    _handleMissingKey(key, value, params) {
        if (this.config.missingHandler.allowEmpty && value === '') {
            return '';
        }
        if (!this.isResolvedMissingOnce && this.useFallbackTranslation()) {
            // We need to set it to true to prevent a loop
            this.isResolvedMissingOnce = true;
            const fallbackValue = this.translate(key, params, this.firstFallbackLang);
            this.isResolvedMissingOnce = false;
            return fallbackValue;
        }
        return this.missingHandler.handle(key, this.getMissingHandlerData(), params);
    }
    /**
     * @internal
     */
    _isLangScoped(lang) {
        return this.getAvailableLangsIds().indexOf(lang) === -1;
    }
    /**
     * Checks if a given string is one of the specified available languages.
     * @returns
     * True if the given string is an available language.
     * False if the given string is not an available language.
     */
    isLang(lang) {
        return this.getAvailableLangsIds().indexOf(lang) !== -1;
    }
    /**
     * @internal
     *
     * We always want to make sure the global lang is loaded
     * before loading the scope since you can access both via the pipe/directive.
     */
    _loadDependencies(path, inlineLoader) {
        const mainLang = getLangFromScope(path);
        if (this._isLangScoped(path) && !this.isLoadedTranslation(mainLang)) {
            return combineLatest(this.load(mainLang), this.load(path, { inlineLoader }));
        }
        return this.load(path, { inlineLoader });
    }
    /**
     * @internal
     */
    _completeScopeWithLang(langOrScope) {
        if (this._isLangScoped(langOrScope) &&
            !this.isLang(getLangFromScope(langOrScope))) {
            return `${langOrScope}/${this.getActiveLang()}`;
        }
        return langOrScope;
    }
    /**
     * @internal
     */
    _setScopeAlias(scope, alias) {
        if (!this.mergedConfig.scopeMapping) {
            this.mergedConfig.scopeMapping = {};
        }
        this.mergedConfig.scopeMapping[scope] = alias;
    }
    ngOnDestroy() {
        if (this.subscription) {
            this.subscription.unsubscribe();
            // Caretaker note: it's important to clean up references to subscriptions since they save the `next`
            // callback within its `destination` property, preventing classes from being GC'd.
            this.subscription = null;
        }
        // Caretaker note: since this is the root provider, it'll be destroyed when the `NgModuleRef.destroy()` is run.
        // Cached values capture `this`, thus leading to a circular reference and preventing the `TranslocoService` from
        // being GC'd. This would lead to a memory leak when server-side rendering is used since the service is created
        // and destroyed per each HTTP request, but any service is not getting GC'd.
        this.cache.clear();
    }
    isLoadedTranslation(lang) {
        return size(this.getTranslation(lang));
    }
    getAvailableLangsIds() {
        const first = this.getAvailableLangs()[0];
        if (isString(first)) {
            return this.getAvailableLangs();
        }
        return this.getAvailableLangs().map((l) => l.id);
    }
    getMissingHandlerData() {
        return {
            ...this.config,
            activeLang: this.getActiveLang(),
            availableLangs: this.availableLangs,
            defaultLang: this.defaultLang,
        };
    }
    /**
     * Use a fallback translation set for missing keys of the primary language
     * This is unrelated to the fallback language (which changes the active language)
     */
    useFallbackTranslation(lang) {
        return (this.config.missingHandler.useFallbackTranslation &&
            lang !== this.firstFallbackLang);
    }
    handleSuccess(lang, translation) {
        this.setTranslation(translation, lang, { emitChange: false });
        this.events.next({
            wasFailure: !!this.failedLangs.size,
            type: 'translationLoadSuccess',
            payload: getEventPayload(lang),
        });
        this.failedLangs.forEach((l) => this.cache.delete(l));
        this.failedLangs.clear();
    }
    handleFailure(lang, loadOptions) {
        // When starting to load a first choice language, initialize
        // the failed counter and resolve the fallback langs.
        if (isNil(loadOptions.failedCounter)) {
            loadOptions.failedCounter = 0;
            if (!loadOptions.fallbackLangs) {
                loadOptions.fallbackLangs = this.fallbackStrategy.getNextLangs(lang);
            }
        }
        const splitted = lang.split('/');
        const fallbacks = loadOptions.fallbackLangs;
        const nextLang = fallbacks[loadOptions.failedCounter];
        this.failedLangs.add(lang);
        // This handles the case where a loaded fallback language is requested again
        if (this.cache.has(nextLang)) {
            this.handleSuccess(nextLang, this.getTranslation(nextLang));
            return EMPTY;
        }
        const isFallbackLang = nextLang === splitted[splitted.length - 1];
        if (!nextLang || isFallbackLang) {
            let msg = `Unable to load translation and all the fallback languages`;
            if (splitted.length > 1) {
                msg += `, did you misspelled the scope name?`;
            }
            throw new Error(msg);
        }
        let resolveLang = nextLang;
        // if it's scoped lang
        if (splitted.length > 1) {
            // We need to resolve it to:
            // todos/langNotExists => todos/nextLang
            splitted[splitted.length - 1] = nextLang;
            resolveLang = splitted.join('/');
        }
        loadOptions.failedCounter++;
        this.events.next({
            type: 'translationLoadFailure',
            payload: getEventPayload(lang),
        });
        return this.load(resolveLang, loadOptions);
    }
    getMappedScope(scope) {
        const { scopeMapping = {} } = this.config;
        return scopeMapping[scope] || toCamelCase(scope);
    }
    /**
     * If lang is scope we need to check the following cases:
     * todos/es => in this case we should take `es` as lang
     * todos => in this case we should set the active lang as lang
     */
    resolveLangAndScope(lang) {
        let resolveLang = lang;
        let scope;
        if (this._isLangScoped(lang)) {
            // en for example
            const langFromScope = getLangFromScope(lang);
            // en is lang
            const hasLang = this.isLang(langFromScope);
            // take en
            resolveLang = hasLang ? langFromScope : this.getActiveLang();
            // find the scope
            scope = this.getMappedScope(hasLang ? getScopeFromLang(lang) : lang);
        }
        return { scope, resolveLang };
    }
    getObjectByKey(translation, key) {
        const result = {};
        const prefix = `${key}.`;
        for (const currentKey in translation) {
            if (currentKey.startsWith(prefix)) {
                result[currentKey.replace(prefix, '')] = translation[currentKey];
            }
        }
        return result;
    }
    getEntries(key) {
        return key instanceof Map ? key.entries() : Object.entries(key);
    }
}
TranslocoService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: TranslocoService, deps: [{ token: TRANSLOCO_LOADER, optional: true }, { token: TRANSLOCO_TRANSPILER }, { token: TRANSLOCO_MISSING_HANDLER }, { token: TRANSLOCO_INTERCEPTOR }, { token: TRANSLOCO_CONFIG }, { token: TRANSLOCO_FALLBACK_STRATEGY }], target: i0.ɵɵFactoryTarget.Injectable });
TranslocoService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: TranslocoService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: TranslocoService, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TRANSLOCO_LOADER]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TRANSLOCO_TRANSPILER]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TRANSLOCO_MISSING_HANDLER]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TRANSLOCO_INTERCEPTOR]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TRANSLOCO_CONFIG]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [TRANSLOCO_FALLBACK_STRATEGY]
                }] }]; } });
//# sourceMappingURL=data:application/json;base64,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