import { ChangeDetectorRef, Directive, ElementRef, Inject, Input, Optional, Renderer2, TemplateRef, ViewContainerRef, } from '@angular/core';
import { forkJoin } from 'rxjs';
import { switchMap } from 'rxjs/operators';
import { TemplateHandler } from './template-handler';
import { TRANSLOCO_LANG } from './transloco-lang';
import { TRANSLOCO_LOADING_TEMPLATE } from './transloco-loading-template';
import { TRANSLOCO_SCOPE } from './transloco-scope';
import { TranslocoService } from './transloco.service';
import { listenOrNotOperator, resolveInlineLoader, shouldListenToLangChanges, } from './shared';
import { LangResolver } from './lang-resolver';
import { ScopeResolver } from './scope-resolver';
import * as i0 from "@angular/core";
import * as i1 from "./transloco.service";
export class TranslocoDirective {
    constructor(translocoService, tpl, providerScope, providerLang, providedLoadingTpl, vcr, cdr, host, renderer) {
        this.translocoService = translocoService;
        this.tpl = tpl;
        this.providerScope = providerScope;
        this.providerLang = providerLang;
        this.providedLoadingTpl = providedLoadingTpl;
        this.vcr = vcr;
        this.cdr = cdr;
        this.host = host;
        this.renderer = renderer;
        this.subscription = null;
        this.translationMemo = {};
        this.params = {};
        // Whether we already rendered the view once
        this.initialized = false;
        this.langResolver = new LangResolver();
        this.scopeResolver = new ScopeResolver(this.translocoService);
        this.strategy = this.tpl === null ? 'attribute' : 'structural';
    }
    static ngTemplateContextGuard(dir, ctx) {
        return true;
    }
    ngOnInit() {
        const listenToLangChange = shouldListenToLangChanges(this.translocoService, this.providerLang || this.inlineLang);
        this.subscription = this.translocoService.langChanges$
            .pipe(switchMap((activeLang) => {
            const lang = this.langResolver.resolve({
                inline: this.inlineLang,
                provider: this.providerLang,
                active: activeLang,
            });
            return Array.isArray(this.providerScope)
                ? forkJoin(this.providerScope.map((providerScope) => this.resolveScope(lang, providerScope)))
                : this.resolveScope(lang, this.providerScope);
        }), listenOrNotOperator(listenToLangChange))
            .subscribe(() => {
            this.currentLang = this.langResolver.resolveLangBasedOnScope(this.path);
            this.strategy === 'attribute'
                ? this.attributeStrategy()
                : this.structuralStrategy(this.currentLang, this.inlineRead);
            this.cdr.markForCheck();
            this.initialized = true;
        });
        const loadingTpl = this.getLoadingTpl();
        if (!this.initialized && loadingTpl) {
            this.loaderTplHandler = new TemplateHandler(loadingTpl, this.vcr);
            this.loaderTplHandler.attachView();
        }
    }
    ngOnChanges(changes) {
        // We need to support dynamic keys/params, so if this is not the first change CD cycle
        // we need to run the function again in order to update the value
        if (this.strategy === 'attribute') {
            const notInit = Object.keys(changes).some((v) => !changes[v].firstChange);
            notInit && this.attributeStrategy();
        }
    }
    attributeStrategy() {
        this.detachLoader();
        this.renderer.setProperty(this.host.nativeElement, 'innerText', this.translocoService.translate(this.key, this.params, this.currentLang));
    }
    structuralStrategy(lang, read) {
        this.translationMemo = {};
        if (this.view) {
            // when the lang changes we need to change the reference so Angular will update the view
            this.view.context['$implicit'] = this.getTranslateFn(lang, read);
            this.view.context['currentLang'] = this.currentLang;
        }
        else {
            this.detachLoader();
            this.view = this.vcr.createEmbeddedView(this.tpl, {
                $implicit: this.getTranslateFn(lang, read),
                currentLang: this.currentLang,
            });
        }
    }
    getTranslateFn(lang, read) {
        return (key, params) => {
            const withRead = read ? `${read}.${key}` : key;
            const withParams = params
                ? `${withRead}${JSON.stringify(params)}`
                : withRead;
            if (Object.prototype.hasOwnProperty.call(this.translationMemo, withParams)) {
                return this.translationMemo[withParams].value;
            }
            this.translationMemo[withParams] = {
                params,
                value: this.translocoService.translate(withRead, params, lang),
            };
            return this.translationMemo[withParams].value;
        };
    }
    getLoadingTpl() {
        return this.inlineTpl || this.providedLoadingTpl;
    }
    ngOnDestroy() {
        if (this.subscription) {
            this.subscription.unsubscribe();
            // Caretaker note: it's important to clean up references to subscriptions since they save the `next`
            // callback within its `destination` property, preventing classes from being GC'd.
            this.subscription = null;
        }
    }
    detachLoader() {
        this.loaderTplHandler?.detachView();
    }
    resolveScope(lang, providerScope) {
        const resolvedScope = this.scopeResolver.resolve({
            inline: this.inlineScope,
            provider: providerScope,
        });
        this.path = this.langResolver.resolveLangPath(lang, resolvedScope);
        const inlineLoader = resolveInlineLoader(providerScope, resolvedScope);
        return this.translocoService._loadDependencies(this.path, inlineLoader);
    }
}
TranslocoDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: TranslocoDirective, deps: [{ token: i1.TranslocoService }, { token: i0.TemplateRef, optional: true }, { token: TRANSLOCO_SCOPE, optional: true }, { token: TRANSLOCO_LANG, optional: true }, { token: TRANSLOCO_LOADING_TEMPLATE, optional: true }, { token: i0.ViewContainerRef }, { token: i0.ChangeDetectorRef }, { token: i0.ElementRef }, { token: i0.Renderer2 }], target: i0.ɵɵFactoryTarget.Directive });
TranslocoDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "12.0.0", version: "13.3.10", type: TranslocoDirective, selector: "[transloco]", inputs: { key: ["transloco", "key"], params: ["translocoParams", "params"], inlineScope: ["translocoScope", "inlineScope"], inlineRead: ["translocoRead", "inlineRead"], inlineLang: ["translocoLang", "inlineLang"], inlineTpl: ["translocoLoadingTpl", "inlineTpl"] }, usesOnChanges: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.10", ngImport: i0, type: TranslocoDirective, decorators: [{
            type: Directive,
            args: [{
                    selector: '[transloco]',
                }]
        }], ctorParameters: function () { return [{ type: i1.TranslocoService }, { type: i0.TemplateRef, decorators: [{
                    type: Optional
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TRANSLOCO_SCOPE]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TRANSLOCO_LANG]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TRANSLOCO_LOADING_TEMPLATE]
                }] }, { type: i0.ViewContainerRef }, { type: i0.ChangeDetectorRef }, { type: i0.ElementRef }, { type: i0.Renderer2 }]; }, propDecorators: { key: [{
                type: Input,
                args: ['transloco']
            }], params: [{
                type: Input,
                args: ['translocoParams']
            }], inlineScope: [{
                type: Input,
                args: ['translocoScope']
            }], inlineRead: [{
                type: Input,
                args: ['translocoRead']
            }], inlineLang: [{
                type: Input,
                args: ['translocoLang']
            }], inlineTpl: [{
                type: Input,
                args: ['translocoLoadingTpl']
            }] } });
//# sourceMappingURL=data:application/json;base64,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