/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ComponentHarness, parallel, } from '@angular/cdk/testing';
import { MatDividerHarness } from '@angular/material/divider/testing';
import { MatLegacySubheaderHarness } from './list-item-harness-base';
/**
 * Shared behavior among the harnesses for the various `MatList` flavors.
 * @template T A constructor type for a list item harness type used by this list harness.
 * @template C The list item harness type that `T` constructs.
 * @template F The filter type used filter list item harness of type `C`.
 * @docs-private
 * @deprecated Use `class` from `@angular/material/list/testing` instead. See https://material.angular.io/guide/mdc-migration for information about migrating.
 * @breaking-change 17.0.0
 */
export class MatLegacyListHarnessBase extends ComponentHarness {
    /**
     * Gets a list of harnesses representing the items in this list.
     * @param filters Optional filters used to narrow which harnesses are included
     * @return The list of items matching the given filters.
     */
    async getItems(filters) {
        return this.locatorForAll(this._itemHarness.with(filters))();
    }
    /**
     * Gets a list of `ListSection` representing the list items grouped by subheaders. If the list has
     * no subheaders it is represented as a single `ListSection` with an undefined `heading` property.
     * @param filters Optional filters used to narrow which list item harnesses are included
     * @return The list of items matching the given filters, grouped into sections by subheader.
     */
    async getItemsGroupedBySubheader(filters) {
        const listSections = [];
        let currentSection = { items: [] };
        const itemsAndSubheaders = await this.getItemsWithSubheadersAndDividers({
            item: filters,
            divider: false,
        });
        for (const itemOrSubheader of itemsAndSubheaders) {
            if (itemOrSubheader instanceof MatLegacySubheaderHarness) {
                if (currentSection.heading !== undefined || currentSection.items.length) {
                    listSections.push(currentSection);
                }
                currentSection = { heading: itemOrSubheader.getText(), items: [] };
            }
            else {
                currentSection.items.push(itemOrSubheader);
            }
        }
        if (currentSection.heading !== undefined ||
            currentSection.items.length ||
            !listSections.length) {
            listSections.push(currentSection);
        }
        // Concurrently wait for all sections to resolve their heading if present.
        return parallel(() => listSections.map(async (s) => ({ items: s.items, heading: await s.heading })));
    }
    /**
     * Gets a list of sub-lists representing the list items grouped by dividers. If the list has no
     * dividers it is represented as a list with a single sub-list.
     * @param filters Optional filters used to narrow which list item harnesses are included
     * @return The list of items matching the given filters, grouped into sub-lists by divider.
     */
    async getItemsGroupedByDividers(filters) {
        const listSections = [[]];
        const itemsAndDividers = await this.getItemsWithSubheadersAndDividers({
            item: filters,
            subheader: false,
        });
        for (const itemOrDivider of itemsAndDividers) {
            if (itemOrDivider instanceof MatDividerHarness) {
                listSections.push([]);
            }
            else {
                listSections[listSections.length - 1].push(itemOrDivider);
            }
        }
        return listSections;
    }
    async getItemsWithSubheadersAndDividers(filters = {}) {
        const query = [];
        if (filters.item !== false) {
            query.push(this._itemHarness.with(filters.item || {}));
        }
        if (filters.subheader !== false) {
            query.push(MatLegacySubheaderHarness.with(filters.subheader));
        }
        if (filters.divider !== false) {
            query.push(MatDividerHarness.with(filters.divider));
        }
        return this.locatorForAll(...query)();
    }
}
//# sourceMappingURL=data:application/json;base64,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