/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { FocusMonitor, FocusTrapFactory, InteractivityChecker } from '@angular/cdk/a11y';
import { OverlayRef } from '@angular/cdk/overlay';
import { DOCUMENT } from '@angular/common';
import { ChangeDetectionStrategy, Component, ElementRef, EventEmitter, Inject, NgZone, Optional, ViewEncapsulation, } from '@angular/core';
import { MatDialogConfig } from './dialog-config';
import { ANIMATION_MODULE_TYPE } from '@angular/platform-browser/animations';
import { CdkDialogContainer } from '@angular/cdk/dialog';
import { coerceNumberProperty } from '@angular/cdk/coercion';
import * as i0 from "@angular/core";
import * as i1 from "@angular/cdk/a11y";
import * as i2 from "./dialog-config";
import * as i3 from "@angular/cdk/overlay";
import * as i4 from "@angular/cdk/portal";
/** Class added when the dialog is open. */
const OPEN_CLASS = 'mdc-dialog--open';
/** Class added while the dialog is opening. */
const OPENING_CLASS = 'mdc-dialog--opening';
/** Class added while the dialog is closing. */
const CLOSING_CLASS = 'mdc-dialog--closing';
/** Duration of the opening animation in milliseconds. */
export const OPEN_ANIMATION_DURATION = 150;
/** Duration of the closing animation in milliseconds. */
export const CLOSE_ANIMATION_DURATION = 75;
/**
 * Base class for the `MatDialogContainer`. The base class does not implement
 * animations as these are left to implementers of the dialog container.
 */
// tslint:disable-next-line:validate-decorators
class _MatDialogContainerBase extends CdkDialogContainer {
    constructor(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, focusMonitor) {
        super(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, focusMonitor);
        /** Emits when an animation state changes. */
        this._animationStateChanged = new EventEmitter();
    }
    _captureInitialFocus() {
        if (!this._config.delayFocusTrap) {
            this._trapFocus();
        }
    }
    /**
     * Callback for when the open dialog animation has finished. Intended to
     * be called by sub-classes that use different animation implementations.
     */
    _openAnimationDone(totalTime) {
        if (this._config.delayFocusTrap) {
            this._trapFocus();
        }
        this._animationStateChanged.next({ state: 'opened', totalTime });
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: _MatDialogContainerBase, deps: [{ token: i0.ElementRef }, { token: i1.FocusTrapFactory }, { token: DOCUMENT, optional: true }, { token: i2.MatDialogConfig }, { token: i1.InteractivityChecker }, { token: i0.NgZone }, { token: i3.OverlayRef }, { token: i1.FocusMonitor }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "16.0.0", type: _MatDialogContainerBase, selector: "ng-component", usesInheritance: true, ngImport: i0, template: '', isInline: true }); }
}
export { _MatDialogContainerBase };
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: _MatDialogContainerBase, decorators: [{
            type: Component,
            args: [{ template: '' }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: i1.FocusTrapFactory }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i2.MatDialogConfig }, { type: i1.InteractivityChecker }, { type: i0.NgZone }, { type: i3.OverlayRef }, { type: i1.FocusMonitor }]; } });
const TRANSITION_DURATION_PROPERTY = '--mat-dialog-transition-duration';
// TODO(mmalerba): Remove this function after animation durations are required
//  to be numbers.
/**
 * Converts a CSS time string to a number in ms. If the given time is already a
 * number, it is assumed to be in ms.
 */
function parseCssTime(time) {
    if (time == null) {
        return null;
    }
    if (typeof time === 'number') {
        return time;
    }
    if (time.endsWith('ms')) {
        return coerceNumberProperty(time.substring(0, time.length - 2));
    }
    if (time.endsWith('s')) {
        return coerceNumberProperty(time.substring(0, time.length - 1)) * 1000;
    }
    if (time === '0') {
        return 0;
    }
    return null; // anything else is invalid.
}
/**
 * Internal component that wraps user-provided dialog content in a MDC dialog.
 * @docs-private
 */
class MatDialogContainer extends _MatDialogContainerBase {
    constructor(elementRef, focusTrapFactory, document, dialogConfig, checker, ngZone, overlayRef, _animationMode, focusMonitor) {
        super(elementRef, focusTrapFactory, document, dialogConfig, checker, ngZone, overlayRef, focusMonitor);
        this._animationMode = _animationMode;
        /** Whether animations are enabled. */
        this._animationsEnabled = this._animationMode !== 'NoopAnimations';
        /** Host element of the dialog container component. */
        this._hostElement = this._elementRef.nativeElement;
        /** Duration of the dialog open animation. */
        this._openAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.enterAnimationDuration) ?? OPEN_ANIMATION_DURATION
            : 0;
        /** Duration of the dialog close animation. */
        this._closeAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.exitAnimationDuration) ?? CLOSE_ANIMATION_DURATION
            : 0;
        /** Current timer for dialog animations. */
        this._animationTimer = null;
        /**
         * Completes the dialog open by clearing potential animation classes, trapping
         * focus and emitting an opened event.
         */
        this._finishDialogOpen = () => {
            this._clearAnimationClasses();
            this._openAnimationDone(this._openAnimationDuration);
        };
        /**
         * Completes the dialog close by clearing potential animation classes, restoring
         * focus and emitting a closed event.
         */
        this._finishDialogClose = () => {
            this._clearAnimationClasses();
            this._animationStateChanged.emit({ state: 'closed', totalTime: this._closeAnimationDuration });
        };
    }
    _contentAttached() {
        // Delegate to the original dialog-container initialization (i.e. saving the
        // previous element, setting up the focus trap and moving focus to the container).
        super._contentAttached();
        // Note: Usually we would be able to use the MDC dialog foundation here to handle
        // the dialog animation for us, but there are a few reasons why we just leverage
        // their styles and not use the runtime foundation code:
        //   1. Foundation does not allow us to disable animations.
        //   2. Foundation contains unnecessary features we don't need and aren't
        //      tree-shakeable. e.g. background scrim, keyboard event handlers for ESC button.
        //   3. Foundation uses unnecessary timers for animations to work around limitations
        //      in React's `setState` mechanism.
        //      https://github.com/material-components/material-components-web/pull/3682.
        this._startOpenAnimation();
    }
    ngOnDestroy() {
        super.ngOnDestroy();
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
    }
    /** Starts the dialog open animation if enabled. */
    _startOpenAnimation() {
        this._animationStateChanged.emit({ state: 'opening', totalTime: this._openAnimationDuration });
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._openAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            // One would expect that the open class is added once the animation finished, but MDC
            // uses the open class in combination with the opening class to start the animation.
            this._requestAnimationFrame(() => this._hostElement.classList.add(OPENING_CLASS, OPEN_CLASS));
            this._waitForAnimationToComplete(this._openAnimationDuration, this._finishDialogOpen);
        }
        else {
            this._hostElement.classList.add(OPEN_CLASS);
            // Note: We could immediately finish the dialog opening here with noop animations,
            // but we defer until next tick so that consumers can subscribe to `afterOpened`.
            // Executing this immediately would mean that `afterOpened` emits synchronously
            // on `dialog.open` before the consumer had a change to subscribe to `afterOpened`.
            Promise.resolve().then(() => this._finishDialogOpen());
        }
    }
    /**
     * Starts the exit animation of the dialog if enabled. This method is
     * called by the dialog ref.
     */
    _startExitAnimation() {
        this._animationStateChanged.emit({ state: 'closing', totalTime: this._closeAnimationDuration });
        this._hostElement.classList.remove(OPEN_CLASS);
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._openAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            this._requestAnimationFrame(() => this._hostElement.classList.add(CLOSING_CLASS));
            this._waitForAnimationToComplete(this._closeAnimationDuration, this._finishDialogClose);
        }
        else {
            // This subscription to the `OverlayRef#backdropClick` observable in the `DialogRef` is
            // set up before any user can subscribe to the backdrop click. The subscription triggers
            // the dialog close and this method synchronously. If we'd synchronously emit the `CLOSED`
            // animation state event if animations are disabled, the overlay would be disposed
            // immediately and all other subscriptions to `DialogRef#backdropClick` would be silently
            // skipped. We work around this by waiting with the dialog close until the next tick when
            // all subscriptions have been fired as expected. This is not an ideal solution, but
            // there doesn't seem to be any other good way. Alternatives that have been considered:
            //   1. Deferring `DialogRef.close`. This could be a breaking change due to a new microtask.
            //      Also this issue is specific to the MDC implementation where the dialog could
            //      technically be closed synchronously. In the non-MDC one, Angular animations are used
            //      and closing always takes at least a tick.
            //   2. Ensuring that user subscriptions to `backdropClick`, `keydownEvents` in the dialog
            //      ref are first. This would solve the issue, but has the risk of memory leaks and also
            //      doesn't solve the case where consumers call `DialogRef.close` in their subscriptions.
            // Based on the fact that this is specific to the MDC-based implementation of the dialog
            // animations, the defer is applied here.
            Promise.resolve().then(() => this._finishDialogClose());
        }
    }
    /** Clears all dialog animation classes. */
    _clearAnimationClasses() {
        this._hostElement.classList.remove(OPENING_CLASS, CLOSING_CLASS);
    }
    _waitForAnimationToComplete(duration, callback) {
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
        // Note that we want this timer to run inside the NgZone, because we want
        // the related events like `afterClosed` to be inside the zone as well.
        this._animationTimer = setTimeout(callback, duration);
    }
    /** Runs a callback in `requestAnimationFrame`, if available. */
    _requestAnimationFrame(callback) {
        this._ngZone.runOutsideAngular(() => {
            if (typeof requestAnimationFrame === 'function') {
                requestAnimationFrame(callback);
            }
            else {
                callback();
            }
        });
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: MatDialogContainer, deps: [{ token: i0.ElementRef }, { token: i1.FocusTrapFactory }, { token: DOCUMENT, optional: true }, { token: i2.MatDialogConfig }, { token: i1.InteractivityChecker }, { token: i0.NgZone }, { token: i3.OverlayRef }, { token: ANIMATION_MODULE_TYPE, optional: true }, { token: i1.FocusMonitor }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "16.0.0", type: MatDialogContainer, selector: "mat-dialog-container", host: { attributes: { "tabindex": "-1" }, properties: { "attr.aria-modal": "_config.ariaModal", "id": "_config.id", "attr.role": "_config.role", "attr.aria-labelledby": "_config.ariaLabel ? null : _ariaLabelledBy", "attr.aria-label": "_config.ariaLabel", "attr.aria-describedby": "_config.ariaDescribedBy || null", "class._mat-animation-noopable": "!_animationsEnabled" }, classAttribute: "mat-mdc-dialog-container mdc-dialog" }, usesInheritance: true, ngImport: i0, template: "<div class=\"mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet></ng-template>\n  </div>\n</div>\n", styles: [".mdc-elevation-overlay{position:absolute;border-radius:inherit;pointer-events:none;opacity:var(--mdc-elevation-overlay-opacity, 0);transition:opacity 280ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-dialog,.mdc-dialog__scrim{position:fixed;top:0;left:0;align-items:center;justify-content:center;box-sizing:border-box;width:100%;height:100%}.mdc-dialog{display:none;z-index:var(--mdc-dialog-z-index, 7)}.mdc-dialog .mdc-dialog__content{padding:20px 24px 20px 24px}.mdc-dialog .mdc-dialog__surface{min-width:280px}@media(max-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:calc(100vw - 32px)}}@media(min-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:560px}}.mdc-dialog .mdc-dialog__surface{max-height:calc(100% - 32px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-width:none}@media(max-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px;width:560px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 112px)}}@media(max-width: 720px)and (min-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:560px}}@media(max-width: 720px)and (max-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:calc(100vh - 160px)}}@media(max-width: 720px)and (min-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px}}@media(max-width: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-height: 400px),(max-width: 600px),(min-width: 720px)and (max-height: 400px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{height:100%;max-height:100vh;max-width:100vw;width:100vw;border-radius:0}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{order:-1;left:-12px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__header{padding:0 16px 9px;justify-content:flex-start}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__title{margin-left:calc(16px - 2 * 12px)}}@media(min-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 400px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}.mdc-dialog.mdc-dialog__scrim--hidden .mdc-dialog__scrim{opacity:0}.mdc-dialog__scrim{opacity:0;z-index:-1}.mdc-dialog__container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;transform:scale(0.8);opacity:0;pointer-events:none}.mdc-dialog__surface{position:relative;display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;max-width:100%;max-height:100%;pointer-events:auto;overflow-y:auto;outline:0}.mdc-dialog__surface .mdc-elevation-overlay{width:100%;height:100%;top:0;left:0}[dir=rtl] .mdc-dialog__surface,.mdc-dialog__surface[dir=rtl]{text-align:right}@media screen and (forced-colors: active),(-ms-high-contrast: active){.mdc-dialog__surface{outline:2px solid windowText}}.mdc-dialog__surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}@media screen and (forced-colors: active){.mdc-dialog__surface::before{border-color:CanvasText}}@media screen and (-ms-high-contrast: active),screen and (-ms-high-contrast: none){.mdc-dialog__surface::before{content:none}}.mdc-dialog__title{display:block;margin-top:0;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:0 24px 9px}.mdc-dialog__title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mdc-dialog__title,.mdc-dialog__title[dir=rtl]{text-align:right}.mdc-dialog--scrollable .mdc-dialog__title{margin-bottom:1px;padding-bottom:15px}.mdc-dialog--fullscreen .mdc-dialog__header{align-items:baseline;border-bottom:1px solid rgba(0,0,0,0);display:inline-flex;justify-content:space-between;padding:0 24px 9px;z-index:1}@media screen and (forced-colors: active){.mdc-dialog--fullscreen .mdc-dialog__header{border-bottom-color:CanvasText}}.mdc-dialog--fullscreen .mdc-dialog__header .mdc-dialog__close{right:-12px}.mdc-dialog--fullscreen .mdc-dialog__title{margin-bottom:0;padding:0;border-bottom:0}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__title{border-bottom:0;margin-bottom:0}.mdc-dialog--fullscreen .mdc-dialog__close{top:5px}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--fullscreen--titleless .mdc-dialog__close{margin-top:4px}.mdc-dialog--fullscreen--titleless.mdc-dialog--scrollable .mdc-dialog__close{margin-top:0}.mdc-dialog__content{flex-grow:1;box-sizing:border-box;margin:0;overflow:auto}.mdc-dialog__content>:first-child{margin-top:0}.mdc-dialog__content>:last-child{margin-bottom:0}.mdc-dialog__title+.mdc-dialog__content,.mdc-dialog__header+.mdc-dialog__content{padding-top:0}.mdc-dialog--scrollable .mdc-dialog__title+.mdc-dialog__content{padding-top:8px;padding-bottom:8px}.mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:6px 0 0}.mdc-dialog--scrollable .mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:0}.mdc-dialog__actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--stacked .mdc-dialog__actions{flex-direction:column;align-items:flex-end}.mdc-dialog__button{margin-left:8px;margin-right:0;max-width:100%;text-align:right}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{margin-left:0;margin-right:8px}.mdc-dialog__button:first-child{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button:first-child,.mdc-dialog__button:first-child[dir=rtl]{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{text-align:left}.mdc-dialog--stacked .mdc-dialog__button:not(:first-child){margin-top:12px}.mdc-dialog--open,.mdc-dialog--opening,.mdc-dialog--closing{display:flex}.mdc-dialog--opening .mdc-dialog__scrim{transition:opacity 150ms linear}.mdc-dialog--opening .mdc-dialog__container{transition:opacity 75ms linear,transform 150ms 0ms cubic-bezier(0, 0, 0.2, 1)}.mdc-dialog--closing .mdc-dialog__scrim,.mdc-dialog--closing .mdc-dialog__container{transition:opacity 75ms linear}.mdc-dialog--closing .mdc-dialog__container{transform:none}.mdc-dialog--open .mdc-dialog__scrim{opacity:1}.mdc-dialog--open .mdc-dialog__container{transform:none;opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim{opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{transition:opacity 75ms linear}.mdc-dialog--open.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim{transition:opacity 150ms linear}.mdc-dialog__surface-scrim{display:none;opacity:0;position:absolute;width:100%;height:100%;z-index:1}.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{display:block}.mdc-dialog-scroll-lock{overflow:hidden}.mdc-dialog--no-content-padding .mdc-dialog__content{padding:0}.mdc-dialog--sheet .mdc-dialog__container .mdc-dialog__close{right:12px;top:9px;position:absolute;z-index:1}.mdc-dialog__scrim--removed{pointer-events:none}.mdc-dialog__scrim--removed .mdc-dialog__scrim,.mdc-dialog__scrim--removed .mdc-dialog__surface-scrim{display:none}.mat-mdc-dialog-content{max-height:65vh}.mat-mdc-dialog-container{position:static;display:block}.mat-mdc-dialog-container,.mat-mdc-dialog-container .mdc-dialog__container,.mat-mdc-dialog-container .mdc-dialog__surface{max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mat-mdc-dialog-container .mdc-dialog__surface{display:block;width:100%;height:100%}.mat-mdc-dialog-container{--mdc-dialog-container-elevation-shadow:0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12);--mdc-dialog-container-shadow-color:#000;--mdc-dialog-container-shape:4px;--mdc-dialog-container-elevation: var(--mdc-dialog-container-elevation-shadow);outline:0}.mat-mdc-dialog-container .mdc-dialog__surface{background-color:var(--mdc-dialog-container-color, white)}.mat-mdc-dialog-container .mdc-dialog__surface{box-shadow:var(--mdc-dialog-container-elevation, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12))}.mat-mdc-dialog-container .mdc-dialog__surface{border-radius:var(--mdc-dialog-container-shape, 4px)}.mat-mdc-dialog-container .mdc-dialog__title{font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87))}.mat-mdc-dialog-container .mdc-dialog__content{font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6))}.mat-mdc-dialog-container .mdc-dialog__container{transition-duration:var(--mat-dialog-transition-duration, 0ms)}.mat-mdc-dialog-container._mat-animation-noopable .mdc-dialog__container{transition:none}.mat-mdc-dialog-content{display:block}.mat-mdc-dialog-actions{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}"], dependencies: [{ kind: "directive", type: i4.CdkPortalOutlet, selector: "[cdkPortalOutlet]", inputs: ["cdkPortalOutlet"], outputs: ["attached"], exportAs: ["cdkPortalOutlet"] }], changeDetection: i0.ChangeDetectionStrategy.Default, encapsulation: i0.ViewEncapsulation.None }); }
}
export { MatDialogContainer };
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: MatDialogContainer, decorators: [{
            type: Component,
            args: [{ selector: 'mat-dialog-container', encapsulation: ViewEncapsulation.None, changeDetection: ChangeDetectionStrategy.Default, host: {
                        'class': 'mat-mdc-dialog-container mdc-dialog',
                        'tabindex': '-1',
                        '[attr.aria-modal]': '_config.ariaModal',
                        '[id]': '_config.id',
                        '[attr.role]': '_config.role',
                        '[attr.aria-labelledby]': '_config.ariaLabel ? null : _ariaLabelledBy',
                        '[attr.aria-label]': '_config.ariaLabel',
                        '[attr.aria-describedby]': '_config.ariaDescribedBy || null',
                        '[class._mat-animation-noopable]': '!_animationsEnabled',
                    }, template: "<div class=\"mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet></ng-template>\n  </div>\n</div>\n", styles: [".mdc-elevation-overlay{position:absolute;border-radius:inherit;pointer-events:none;opacity:var(--mdc-elevation-overlay-opacity, 0);transition:opacity 280ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-dialog,.mdc-dialog__scrim{position:fixed;top:0;left:0;align-items:center;justify-content:center;box-sizing:border-box;width:100%;height:100%}.mdc-dialog{display:none;z-index:var(--mdc-dialog-z-index, 7)}.mdc-dialog .mdc-dialog__content{padding:20px 24px 20px 24px}.mdc-dialog .mdc-dialog__surface{min-width:280px}@media(max-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:calc(100vw - 32px)}}@media(min-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:560px}}.mdc-dialog .mdc-dialog__surface{max-height:calc(100% - 32px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-width:none}@media(max-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px;width:560px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 112px)}}@media(max-width: 720px)and (min-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:560px}}@media(max-width: 720px)and (max-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:calc(100vh - 160px)}}@media(max-width: 720px)and (min-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px}}@media(max-width: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-height: 400px),(max-width: 600px),(min-width: 720px)and (max-height: 400px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{height:100%;max-height:100vh;max-width:100vw;width:100vw;border-radius:0}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{order:-1;left:-12px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__header{padding:0 16px 9px;justify-content:flex-start}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__title{margin-left:calc(16px - 2 * 12px)}}@media(min-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 400px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}.mdc-dialog.mdc-dialog__scrim--hidden .mdc-dialog__scrim{opacity:0}.mdc-dialog__scrim{opacity:0;z-index:-1}.mdc-dialog__container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;transform:scale(0.8);opacity:0;pointer-events:none}.mdc-dialog__surface{position:relative;display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;max-width:100%;max-height:100%;pointer-events:auto;overflow-y:auto;outline:0}.mdc-dialog__surface .mdc-elevation-overlay{width:100%;height:100%;top:0;left:0}[dir=rtl] .mdc-dialog__surface,.mdc-dialog__surface[dir=rtl]{text-align:right}@media screen and (forced-colors: active),(-ms-high-contrast: active){.mdc-dialog__surface{outline:2px solid windowText}}.mdc-dialog__surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}@media screen and (forced-colors: active){.mdc-dialog__surface::before{border-color:CanvasText}}@media screen and (-ms-high-contrast: active),screen and (-ms-high-contrast: none){.mdc-dialog__surface::before{content:none}}.mdc-dialog__title{display:block;margin-top:0;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:0 24px 9px}.mdc-dialog__title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mdc-dialog__title,.mdc-dialog__title[dir=rtl]{text-align:right}.mdc-dialog--scrollable .mdc-dialog__title{margin-bottom:1px;padding-bottom:15px}.mdc-dialog--fullscreen .mdc-dialog__header{align-items:baseline;border-bottom:1px solid rgba(0,0,0,0);display:inline-flex;justify-content:space-between;padding:0 24px 9px;z-index:1}@media screen and (forced-colors: active){.mdc-dialog--fullscreen .mdc-dialog__header{border-bottom-color:CanvasText}}.mdc-dialog--fullscreen .mdc-dialog__header .mdc-dialog__close{right:-12px}.mdc-dialog--fullscreen .mdc-dialog__title{margin-bottom:0;padding:0;border-bottom:0}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__title{border-bottom:0;margin-bottom:0}.mdc-dialog--fullscreen .mdc-dialog__close{top:5px}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--fullscreen--titleless .mdc-dialog__close{margin-top:4px}.mdc-dialog--fullscreen--titleless.mdc-dialog--scrollable .mdc-dialog__close{margin-top:0}.mdc-dialog__content{flex-grow:1;box-sizing:border-box;margin:0;overflow:auto}.mdc-dialog__content>:first-child{margin-top:0}.mdc-dialog__content>:last-child{margin-bottom:0}.mdc-dialog__title+.mdc-dialog__content,.mdc-dialog__header+.mdc-dialog__content{padding-top:0}.mdc-dialog--scrollable .mdc-dialog__title+.mdc-dialog__content{padding-top:8px;padding-bottom:8px}.mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:6px 0 0}.mdc-dialog--scrollable .mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:0}.mdc-dialog__actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--stacked .mdc-dialog__actions{flex-direction:column;align-items:flex-end}.mdc-dialog__button{margin-left:8px;margin-right:0;max-width:100%;text-align:right}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{margin-left:0;margin-right:8px}.mdc-dialog__button:first-child{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button:first-child,.mdc-dialog__button:first-child[dir=rtl]{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{text-align:left}.mdc-dialog--stacked .mdc-dialog__button:not(:first-child){margin-top:12px}.mdc-dialog--open,.mdc-dialog--opening,.mdc-dialog--closing{display:flex}.mdc-dialog--opening .mdc-dialog__scrim{transition:opacity 150ms linear}.mdc-dialog--opening .mdc-dialog__container{transition:opacity 75ms linear,transform 150ms 0ms cubic-bezier(0, 0, 0.2, 1)}.mdc-dialog--closing .mdc-dialog__scrim,.mdc-dialog--closing .mdc-dialog__container{transition:opacity 75ms linear}.mdc-dialog--closing .mdc-dialog__container{transform:none}.mdc-dialog--open .mdc-dialog__scrim{opacity:1}.mdc-dialog--open .mdc-dialog__container{transform:none;opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim{opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{transition:opacity 75ms linear}.mdc-dialog--open.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim{transition:opacity 150ms linear}.mdc-dialog__surface-scrim{display:none;opacity:0;position:absolute;width:100%;height:100%;z-index:1}.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{display:block}.mdc-dialog-scroll-lock{overflow:hidden}.mdc-dialog--no-content-padding .mdc-dialog__content{padding:0}.mdc-dialog--sheet .mdc-dialog__container .mdc-dialog__close{right:12px;top:9px;position:absolute;z-index:1}.mdc-dialog__scrim--removed{pointer-events:none}.mdc-dialog__scrim--removed .mdc-dialog__scrim,.mdc-dialog__scrim--removed .mdc-dialog__surface-scrim{display:none}.mat-mdc-dialog-content{max-height:65vh}.mat-mdc-dialog-container{position:static;display:block}.mat-mdc-dialog-container,.mat-mdc-dialog-container .mdc-dialog__container,.mat-mdc-dialog-container .mdc-dialog__surface{max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mat-mdc-dialog-container .mdc-dialog__surface{display:block;width:100%;height:100%}.mat-mdc-dialog-container{--mdc-dialog-container-elevation-shadow:0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12);--mdc-dialog-container-shadow-color:#000;--mdc-dialog-container-shape:4px;--mdc-dialog-container-elevation: var(--mdc-dialog-container-elevation-shadow);outline:0}.mat-mdc-dialog-container .mdc-dialog__surface{background-color:var(--mdc-dialog-container-color, white)}.mat-mdc-dialog-container .mdc-dialog__surface{box-shadow:var(--mdc-dialog-container-elevation, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12))}.mat-mdc-dialog-container .mdc-dialog__surface{border-radius:var(--mdc-dialog-container-shape, 4px)}.mat-mdc-dialog-container .mdc-dialog__title{font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87))}.mat-mdc-dialog-container .mdc-dialog__content{font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6))}.mat-mdc-dialog-container .mdc-dialog__container{transition-duration:var(--mat-dialog-transition-duration, 0ms)}.mat-mdc-dialog-container._mat-animation-noopable .mdc-dialog__container{transition:none}.mat-mdc-dialog-content{display:block}.mat-mdc-dialog-actions{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}"] }]
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: i1.FocusTrapFactory }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i2.MatDialogConfig }, { type: i1.InteractivityChecker }, { type: i0.NgZone }, { type: i3.OverlayRef }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [ANIMATION_MODULE_TYPE]
                }] }, { type: i1.FocusMonitor }]; } });
//# sourceMappingURL=data:application/json;base64,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