/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Inject, Injectable, Optional, InjectionToken } from '@angular/core';
import { DateAdapter, MAT_DATE_LOCALE } from '@angular/material/core';
import { DateTime as LuxonDateTime, Info as LuxonInfo, } from 'luxon';
import * as i0 from "@angular/core";
/** InjectionToken for LuxonDateAdapter to configure options. */
export const MAT_LUXON_DATE_ADAPTER_OPTIONS = new InjectionToken('MAT_LUXON_DATE_ADAPTER_OPTIONS', {
    providedIn: 'root',
    factory: MAT_LUXON_DATE_ADAPTER_OPTIONS_FACTORY,
});
/** @docs-private */
export function MAT_LUXON_DATE_ADAPTER_OPTIONS_FACTORY() {
    return {
        useUtc: false,
        firstDayOfWeek: 0,
    };
}
/** Creates an array and fills it with values. */
function range(length, valueFunction) {
    const valuesArray = Array(length);
    for (let i = 0; i < length; i++) {
        valuesArray[i] = valueFunction(i);
    }
    return valuesArray;
}
/** Adapts Luxon Dates for use with Angular Material. */
class LuxonDateAdapter extends DateAdapter {
    constructor(dateLocale, options) {
        super();
        this._useUTC = !!options?.useUtc;
        this._firstDayOfWeek = options?.firstDayOfWeek || 0;
        this.setLocale(dateLocale || LuxonDateTime.local().locale);
    }
    getYear(date) {
        return date.year;
    }
    getMonth(date) {
        // Luxon works with 1-indexed months whereas our code expects 0-indexed.
        return date.month - 1;
    }
    getDate(date) {
        return date.day;
    }
    getDayOfWeek(date) {
        return date.weekday;
    }
    getMonthNames(style) {
        return LuxonInfo.months(style, { locale: this.locale });
    }
    getDateNames() {
        // At the time of writing, Luxon doesn't offer similar
        // functionality so we have to fall back to the Intl API.
        const dtf = new Intl.DateTimeFormat(this.locale, { day: 'numeric', timeZone: 'utc' });
        // Format a UTC date in order to avoid DST issues.
        return range(31, i => dtf.format(LuxonDateTime.utc(2017, 1, i + 1).toJSDate()));
    }
    getDayOfWeekNames(style) {
        // Note that we shift the array once, because Luxon returns Monday as the
        // first day of the week, whereas our logic assumes that it's Sunday. See:
        // https://moment.github.io/luxon/api-docs/index.html#infoweekdays
        const days = LuxonInfo.weekdays(style, { locale: this.locale });
        days.unshift(days.pop());
        return days;
    }
    getYearName(date) {
        return date.toFormat('yyyy');
    }
    getFirstDayOfWeek() {
        return this._firstDayOfWeek;
    }
    getNumDaysInMonth(date) {
        return date.daysInMonth;
    }
    clone(date) {
        return LuxonDateTime.fromObject(date.toObject());
    }
    createDate(year, month, date) {
        if (month < 0 || month > 11) {
            throw Error(`Invalid month index "${month}". Month index has to be between 0 and 11.`);
        }
        if (date < 1) {
            throw Error(`Invalid date "${date}". Date has to be greater than 0.`);
        }
        // Luxon uses 1-indexed months so we need to add one to the month.
        const result = this._useUTC
            ? LuxonDateTime.utc(year, month + 1, date)
            : LuxonDateTime.local(year, month + 1, date);
        if (!this.isValid(result)) {
            throw Error(`Invalid date "${date}". Reason: "${result.invalidReason}".`);
        }
        return result.setLocale(this.locale);
    }
    today() {
        return (this._useUTC ? LuxonDateTime.utc() : LuxonDateTime.local()).setLocale(this.locale);
    }
    parse(value, parseFormat) {
        const options = this._getOptions();
        if (typeof value == 'string' && value.length > 0) {
            const iso8601Date = LuxonDateTime.fromISO(value, options);
            if (this.isValid(iso8601Date)) {
                return iso8601Date;
            }
            const formats = Array.isArray(parseFormat) ? parseFormat : [parseFormat];
            if (!parseFormat.length) {
                throw Error('Formats array must not be empty.');
            }
            for (const format of formats) {
                const fromFormat = LuxonDateTime.fromFormat(value, format, options);
                if (this.isValid(fromFormat)) {
                    return fromFormat;
                }
            }
            return this.invalid();
        }
        else if (typeof value === 'number') {
            return LuxonDateTime.fromMillis(value, options);
        }
        else if (value instanceof Date) {
            return LuxonDateTime.fromJSDate(value, options);
        }
        else if (value instanceof LuxonDateTime) {
            return LuxonDateTime.fromMillis(value.toMillis(), options);
        }
        return null;
    }
    format(date, displayFormat) {
        if (!this.isValid(date)) {
            throw Error('LuxonDateAdapter: Cannot format invalid date.');
        }
        if (this._useUTC) {
            return date.setLocale(this.locale).setZone('utc').toFormat(displayFormat);
        }
        else {
            return date.setLocale(this.locale).toFormat(displayFormat);
        }
    }
    addCalendarYears(date, years) {
        return date.plus({ years }).setLocale(this.locale);
    }
    addCalendarMonths(date, months) {
        return date.plus({ months }).setLocale(this.locale);
    }
    addCalendarDays(date, days) {
        return date.plus({ days }).setLocale(this.locale);
    }
    toIso8601(date) {
        return date.toISO();
    }
    /**
     * Returns the given value if given a valid Luxon or null. Deserializes valid ISO 8601 strings
     * (https://www.ietf.org/rfc/rfc3339.txt) and valid Date objects into valid DateTime and empty
     * string into null. Returns an invalid date for all other values.
     */
    deserialize(value) {
        const options = this._getOptions();
        let date;
        if (value instanceof Date) {
            date = LuxonDateTime.fromJSDate(value, options);
        }
        if (typeof value === 'string') {
            if (!value) {
                return null;
            }
            date = LuxonDateTime.fromISO(value, options);
        }
        if (date && this.isValid(date)) {
            return date;
        }
        return super.deserialize(value);
    }
    isDateInstance(obj) {
        return obj instanceof LuxonDateTime;
    }
    isValid(date) {
        return date.isValid;
    }
    invalid() {
        return LuxonDateTime.invalid('Invalid Luxon DateTime object.');
    }
    /** Gets the options that should be used when constructing a new `DateTime` object. */
    _getOptions() {
        return {
            zone: this._useUTC ? 'utc' : undefined,
            locale: this.locale,
        };
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: LuxonDateAdapter, deps: [{ token: MAT_DATE_LOCALE, optional: true }, { token: MAT_LUXON_DATE_ADAPTER_OPTIONS, optional: true }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: LuxonDateAdapter }); }
}
export { LuxonDateAdapter };
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "16.0.0", ngImport: i0, type: LuxonDateAdapter, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [MAT_DATE_LOCALE]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [MAT_LUXON_DATE_ADAPTER_OPTIONS]
                }] }]; } });
//# sourceMappingURL=data:application/json;base64,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