/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Directive, Inject, Injectable, Input } from '@angular/core';
import { BaseDirective2, LAYOUT_CONFIG, validateBasis, StyleBuilder, } from '@angular/flex-layout/core';
import { takeUntil } from 'rxjs/operators';
import { extendObject } from '@angular/flex-layout/_private-utils';
import { isFlowHorizontal } from '@angular/flex-layout/_private-utils';
import * as i0 from "@angular/core";
import * as i1 from "@angular/flex-layout/core";
export class FlexStyleBuilder extends StyleBuilder {
    constructor(layoutConfig) {
        super();
        this.layoutConfig = layoutConfig;
    }
    buildStyles(input, parent) {
        let [grow, shrink, ...basisParts] = input.split(' ');
        let basis = basisParts.join(' ');
        // The flex-direction of this element's flex container. Defaults to 'row'.
        const direction = (parent.direction.indexOf('column') > -1) ? 'column' : 'row';
        const max = isFlowHorizontal(direction) ? 'max-width' : 'max-height';
        const min = isFlowHorizontal(direction) ? 'min-width' : 'min-height';
        const hasCalc = String(basis).indexOf('calc') > -1;
        const usingCalc = hasCalc || (basis === 'auto');
        const isPercent = String(basis).indexOf('%') > -1 && !hasCalc;
        const hasUnits = String(basis).indexOf('px') > -1 || String(basis).indexOf('rem') > -1 ||
            String(basis).indexOf('em') > -1 || String(basis).indexOf('vw') > -1 ||
            String(basis).indexOf('vh') > -1;
        let isValue = (hasCalc || hasUnits);
        grow = (grow == '0') ? 0 : grow;
        shrink = (shrink == '0') ? 0 : shrink;
        // make box inflexible when shrink and grow are both zero
        // should not set a min when the grow is zero
        // should not set a max when the shrink is zero
        const isFixed = !grow && !shrink;
        let css = {};
        // flex-basis allows you to specify the initial/starting main-axis size of the element,
        // before anything else is computed. It can either be a percentage or an absolute value.
        // It is, however, not the breaking point for flex-grow/shrink properties
        //
        // flex-grow can be seen as this:
        //   0: Do not stretch. Either size to element's content width, or obey 'flex-basis'.
        //   1: (Default value). Stretch; will be the same size to all other flex items on
        //       the same row since they have a default value of 1.
        //   ≥2 (integer n): Stretch. Will be n times the size of other elements
        //      with 'flex-grow: 1' on the same row.
        // Use `null` to clear existing styles.
        const clearStyles = {
            'max-width': null,
            'max-height': null,
            'min-width': null,
            'min-height': null
        };
        switch (basis || '') {
            case '':
                const useColumnBasisZero = this.layoutConfig.useColumnBasisZero !== false;
                basis = direction === 'row' ? '0%' : (useColumnBasisZero ? '0.000000001px' : 'auto');
                break;
            case 'initial': // default
            case 'nogrow':
                grow = 0;
                basis = 'auto';
                break;
            case 'grow':
                basis = '100%';
                break;
            case 'noshrink':
                shrink = 0;
                basis = 'auto';
                break;
            case 'auto':
                break;
            case 'none':
                grow = 0;
                shrink = 0;
                basis = 'auto';
                break;
            default:
                // Defaults to percentage sizing unless `px` is explicitly set
                if (!isValue && !isPercent && !isNaN(basis)) {
                    basis = basis + '%';
                }
                // Fix for issue 280
                if (basis === '0%') {
                    isValue = true;
                }
                if (basis === '0px') {
                    basis = '0%';
                }
                // fix issue #5345
                if (hasCalc) {
                    css = extendObject(clearStyles, {
                        'flex-grow': grow,
                        'flex-shrink': shrink,
                        'flex-basis': isValue ? basis : '100%'
                    });
                }
                else {
                    css = extendObject(clearStyles, {
                        'flex': `${grow} ${shrink} ${isValue ? basis : '100%'}`
                    });
                }
                break;
        }
        if (!(css['flex'] || css['flex-grow'])) {
            if (hasCalc) {
                css = extendObject(clearStyles, {
                    'flex-grow': grow,
                    'flex-shrink': shrink,
                    'flex-basis': basis
                });
            }
            else {
                css = extendObject(clearStyles, {
                    'flex': `${grow} ${shrink} ${basis}`
                });
            }
        }
        // Fix for issues 277, 534, and 728
        if (basis !== '0%' && basis !== '0px' && basis !== '0.000000001px' && basis !== 'auto') {
            css[min] = isFixed || (isValue && grow) ? basis : null;
            css[max] = isFixed || (!usingCalc && shrink) ? basis : null;
        }
        // Fix for issue 528
        if (!css[min] && !css[max]) {
            if (hasCalc) {
                css = extendObject(clearStyles, {
                    'flex-grow': grow,
                    'flex-shrink': shrink,
                    'flex-basis': basis
                });
            }
            else {
                css = extendObject(clearStyles, {
                    'flex': `${grow} ${shrink} ${basis}`
                });
            }
        }
        else {
            // Fix for issue 660
            if (parent.hasWrap) {
                css[hasCalc ? 'flex-basis' : 'flex'] = css[max] ?
                    (hasCalc ? css[max] : `${grow} ${shrink} ${css[max]}`) :
                    (hasCalc ? css[min] : `${grow} ${shrink} ${css[min]}`);
            }
        }
        return extendObject(css, { 'box-sizing': 'border-box' });
    }
}
FlexStyleBuilder.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: FlexStyleBuilder, deps: [{ token: LAYOUT_CONFIG }], target: i0.ɵɵFactoryTarget.Injectable });
FlexStyleBuilder.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: FlexStyleBuilder, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: FlexStyleBuilder, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [LAYOUT_CONFIG]
                }] }]; } });
const inputs = [
    'fxFlex', 'fxFlex.xs', 'fxFlex.sm', 'fxFlex.md',
    'fxFlex.lg', 'fxFlex.xl', 'fxFlex.lt-sm', 'fxFlex.lt-md',
    'fxFlex.lt-lg', 'fxFlex.lt-xl', 'fxFlex.gt-xs', 'fxFlex.gt-sm',
    'fxFlex.gt-md', 'fxFlex.gt-lg'
];
const selector = `
  [fxFlex], [fxFlex.xs], [fxFlex.sm], [fxFlex.md],
  [fxFlex.lg], [fxFlex.xl], [fxFlex.lt-sm], [fxFlex.lt-md],
  [fxFlex.lt-lg], [fxFlex.lt-xl], [fxFlex.gt-xs], [fxFlex.gt-sm],
  [fxFlex.gt-md], [fxFlex.gt-lg]
`;
/**
 * Directive to control the size of a flex item using flex-basis, flex-grow, and flex-shrink.
 * Corresponds to the css `flex` shorthand property.
 *
 * @see https://css-tricks.com/snippets/css/a-guide-to-flexbox/
 */
export class FlexDirective extends BaseDirective2 {
    constructor(elRef, styleUtils, layoutConfig, styleBuilder, marshal) {
        super(elRef, styleBuilder, styleUtils, marshal);
        this.layoutConfig = layoutConfig;
        this.marshal = marshal;
        this.DIRECTIVE_KEY = 'flex';
        this.direction = undefined;
        this.wrap = undefined;
        this.flexGrow = '1';
        this.flexShrink = '1';
        this.init();
    }
    get shrink() { return this.flexShrink; }
    set shrink(value) {
        this.flexShrink = value || '1';
        this.triggerReflow();
    }
    get grow() { return this.flexGrow; }
    set grow(value) {
        this.flexGrow = value || '1';
        this.triggerReflow();
    }
    ngOnInit() {
        if (this.parentElement) {
            this.marshal.trackValue(this.parentElement, 'layout')
                .pipe(takeUntil(this.destroySubject))
                .subscribe(this.onLayoutChange.bind(this));
            this.marshal.trackValue(this.nativeElement, 'layout-align')
                .pipe(takeUntil(this.destroySubject))
                .subscribe(this.triggerReflow.bind(this));
        }
    }
    /**
     * Caches the parent container's 'flex-direction' and updates the element's style.
     * Used as a handler for layout change events from the parent flex container.
     */
    onLayoutChange(matcher) {
        const layout = matcher.value;
        const layoutParts = layout.split(' ');
        this.direction = layoutParts[0];
        this.wrap = layoutParts[1] !== undefined && layoutParts[1] === 'wrap';
        this.triggerUpdate();
    }
    /** Input to this is exclusively the basis input value */
    updateWithValue(value) {
        const addFlexToParent = this.layoutConfig.addFlexToParent !== false;
        if (this.direction === undefined) {
            this.direction = this.getFlexFlowDirection(this.parentElement, addFlexToParent);
        }
        if (this.wrap === undefined) {
            this.wrap = this.hasWrap(this.parentElement);
        }
        const direction = this.direction;
        const isHorizontal = direction.startsWith('row');
        const hasWrap = this.wrap;
        if (isHorizontal && hasWrap) {
            this.styleCache = flexRowWrapCache;
        }
        else if (isHorizontal && !hasWrap) {
            this.styleCache = flexRowCache;
        }
        else if (!isHorizontal && hasWrap) {
            this.styleCache = flexColumnWrapCache;
        }
        else if (!isHorizontal && !hasWrap) {
            this.styleCache = flexColumnCache;
        }
        const basis = String(value).replace(';', '');
        const parts = validateBasis(basis, this.flexGrow, this.flexShrink);
        this.addStyles(parts.join(' '), { direction, hasWrap });
    }
    /** Trigger a style reflow, usually based on a shrink/grow input event */
    triggerReflow() {
        const activatedValue = this.activatedValue;
        if (activatedValue !== undefined) {
            const parts = validateBasis(activatedValue + '', this.flexGrow, this.flexShrink);
            this.marshal.updateElement(this.nativeElement, this.DIRECTIVE_KEY, parts.join(' '));
        }
    }
}
FlexDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: FlexDirective, deps: [{ token: i0.ElementRef }, { token: i1.StyleUtils }, { token: LAYOUT_CONFIG }, { token: FlexStyleBuilder }, { token: i1.MediaMarshaller }], target: i0.ɵɵFactoryTarget.Directive });
FlexDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "14.0.0", version: "15.0.2", type: FlexDirective, inputs: { shrink: ["fxShrink", "shrink"], grow: ["fxGrow", "grow"] }, usesInheritance: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: FlexDirective, decorators: [{
            type: Directive
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: i1.StyleUtils }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [LAYOUT_CONFIG]
                }] }, { type: FlexStyleBuilder }, { type: i1.MediaMarshaller }]; }, propDecorators: { shrink: [{
                type: Input,
                args: ['fxShrink']
            }], grow: [{
                type: Input,
                args: ['fxGrow']
            }] } });
export class DefaultFlexDirective extends FlexDirective {
    constructor() {
        super(...arguments);
        this.inputs = inputs;
    }
}
DefaultFlexDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: DefaultFlexDirective, deps: null, target: i0.ɵɵFactoryTarget.Directive });
DefaultFlexDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "14.0.0", version: "15.0.2", type: DefaultFlexDirective, selector: "\n  [fxFlex], [fxFlex.xs], [fxFlex.sm], [fxFlex.md],\n  [fxFlex.lg], [fxFlex.xl], [fxFlex.lt-sm], [fxFlex.lt-md],\n  [fxFlex.lt-lg], [fxFlex.lt-xl], [fxFlex.gt-xs], [fxFlex.gt-sm],\n  [fxFlex.gt-md], [fxFlex.gt-lg]\n", inputs: { fxFlex: "fxFlex", "fxFlex.xs": "fxFlex.xs", "fxFlex.sm": "fxFlex.sm", "fxFlex.md": "fxFlex.md", "fxFlex.lg": "fxFlex.lg", "fxFlex.xl": "fxFlex.xl", "fxFlex.lt-sm": "fxFlex.lt-sm", "fxFlex.lt-md": "fxFlex.lt-md", "fxFlex.lt-lg": "fxFlex.lt-lg", "fxFlex.lt-xl": "fxFlex.lt-xl", "fxFlex.gt-xs": "fxFlex.gt-xs", "fxFlex.gt-sm": "fxFlex.gt-sm", "fxFlex.gt-md": "fxFlex.gt-md", "fxFlex.gt-lg": "fxFlex.gt-lg" }, usesInheritance: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: DefaultFlexDirective, decorators: [{
            type: Directive,
            args: [{ inputs, selector }]
        }] });
const flexRowCache = new Map();
const flexColumnCache = new Map();
const flexRowWrapCache = new Map();
const flexColumnWrapCache = new Map();
//# sourceMappingURL=data:application/json;base64,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