/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Directive, Inject, Optional, PLATFORM_ID, SecurityContext, Self, } from '@angular/core';
import { isPlatformServer, NgStyle } from '@angular/common';
import { BaseDirective2, SERVER_TOKEN, } from '@angular/flex-layout/core';
import { buildRawList, getType, buildMapFromSet, stringToKeyValue, keyValuesToMap, } from './style-transforms';
import * as i0 from "@angular/core";
import * as i1 from "@angular/flex-layout/core";
import * as i2 from "@angular/platform-browser";
import * as i3 from "@angular/common";
export class StyleDirective extends BaseDirective2 {
    constructor(elementRef, styler, marshal, sanitizer, differs, renderer2, ngStyleInstance, serverLoaded, platformId) {
        super(elementRef, null, styler, marshal);
        this.sanitizer = sanitizer;
        this.ngStyleInstance = ngStyleInstance;
        this.DIRECTIVE_KEY = 'ngStyle';
        if (!this.ngStyleInstance) {
            // Create an instance NgStyle Directive instance only if `ngStyle=""` has NOT been
            // defined on the same host element; since the responsive variations may be defined...
            this.ngStyleInstance = new NgStyle(elementRef, differs, renderer2);
        }
        this.init();
        const styles = this.nativeElement.getAttribute('style') ?? '';
        this.fallbackStyles = this.buildStyleMap(styles);
        this.isServer = serverLoaded && isPlatformServer(platformId);
    }
    /** Add generated styles */
    updateWithValue(value) {
        const styles = this.buildStyleMap(value);
        this.ngStyleInstance.ngStyle = { ...this.fallbackStyles, ...styles };
        if (this.isServer) {
            this.applyStyleToElement(styles);
        }
        this.ngStyleInstance.ngDoCheck();
    }
    /** Remove generated styles */
    clearStyles() {
        this.ngStyleInstance.ngStyle = this.fallbackStyles;
        this.ngStyleInstance.ngDoCheck();
    }
    /**
     * Convert raw strings to ngStyleMap; which is required by ngStyle
     * NOTE: Raw string key-value pairs MUST be delimited by `;`
     *       Comma-delimiters are not supported due to complexities of
     *       possible style values such as `rgba(x,x,x,x)` and others
     */
    buildStyleMap(styles) {
        // Always safe-guard (aka sanitize) style property values
        const sanitizer = (val) => this.sanitizer.sanitize(SecurityContext.STYLE, val) ?? '';
        if (styles) {
            switch (getType(styles)) {
                case 'string': return buildMapFromList(buildRawList(styles), sanitizer);
                case 'array': return buildMapFromList(styles, sanitizer);
                case 'set': return buildMapFromSet(styles, sanitizer);
                default: return buildMapFromSet(styles, sanitizer);
            }
        }
        return {};
    }
    // ******************************************************************
    // Lifecycle Hooks
    // ******************************************************************
    /** For ChangeDetectionStrategy.onPush and ngOnChanges() updates */
    ngDoCheck() {
        this.ngStyleInstance.ngDoCheck();
    }
}
StyleDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: StyleDirective, deps: [{ token: i0.ElementRef }, { token: i1.StyleUtils }, { token: i1.MediaMarshaller }, { token: i2.DomSanitizer }, { token: i0.KeyValueDiffers }, { token: i0.Renderer2 }, { token: i3.NgStyle, optional: true, self: true }, { token: SERVER_TOKEN }, { token: PLATFORM_ID }], target: i0.ɵɵFactoryTarget.Directive });
StyleDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "14.0.0", version: "15.0.2", type: StyleDirective, usesInheritance: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: StyleDirective, decorators: [{
            type: Directive
        }], ctorParameters: function () { return [{ type: i0.ElementRef }, { type: i1.StyleUtils }, { type: i1.MediaMarshaller }, { type: i2.DomSanitizer }, { type: i0.KeyValueDiffers }, { type: i0.Renderer2 }, { type: i3.NgStyle, decorators: [{
                    type: Optional
                }, {
                    type: Self
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [SERVER_TOKEN]
                }] }, { type: Object, decorators: [{
                    type: Inject,
                    args: [PLATFORM_ID]
                }] }]; } });
const inputs = [
    'ngStyle',
    'ngStyle.xs', 'ngStyle.sm', 'ngStyle.md', 'ngStyle.lg', 'ngStyle.xl',
    'ngStyle.lt-sm', 'ngStyle.lt-md', 'ngStyle.lt-lg', 'ngStyle.lt-xl',
    'ngStyle.gt-xs', 'ngStyle.gt-sm', 'ngStyle.gt-md', 'ngStyle.gt-lg'
];
const selector = `
  [ngStyle],
  [ngStyle.xs], [ngStyle.sm], [ngStyle.md], [ngStyle.lg], [ngStyle.xl],
  [ngStyle.lt-sm], [ngStyle.lt-md], [ngStyle.lt-lg], [ngStyle.lt-xl],
  [ngStyle.gt-xs], [ngStyle.gt-sm], [ngStyle.gt-md], [ngStyle.gt-lg]
`;
/**
 * Directive to add responsive support for ngStyle.
 *
 */
export class DefaultStyleDirective extends StyleDirective {
    constructor() {
        super(...arguments);
        this.inputs = inputs;
    }
}
DefaultStyleDirective.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: DefaultStyleDirective, deps: null, target: i0.ɵɵFactoryTarget.Directive });
DefaultStyleDirective.ɵdir = i0.ɵɵngDeclareDirective({ minVersion: "14.0.0", version: "15.0.2", type: DefaultStyleDirective, selector: "\n  [ngStyle],\n  [ngStyle.xs], [ngStyle.sm], [ngStyle.md], [ngStyle.lg], [ngStyle.xl],\n  [ngStyle.lt-sm], [ngStyle.lt-md], [ngStyle.lt-lg], [ngStyle.lt-xl],\n  [ngStyle.gt-xs], [ngStyle.gt-sm], [ngStyle.gt-md], [ngStyle.gt-lg]\n", inputs: { ngStyle: "ngStyle", "ngStyle.xs": "ngStyle.xs", "ngStyle.sm": "ngStyle.sm", "ngStyle.md": "ngStyle.md", "ngStyle.lg": "ngStyle.lg", "ngStyle.xl": "ngStyle.xl", "ngStyle.lt-sm": "ngStyle.lt-sm", "ngStyle.lt-md": "ngStyle.lt-md", "ngStyle.lt-lg": "ngStyle.lt-lg", "ngStyle.lt-xl": "ngStyle.lt-xl", "ngStyle.gt-xs": "ngStyle.gt-xs", "ngStyle.gt-sm": "ngStyle.gt-sm", "ngStyle.gt-md": "ngStyle.gt-md", "ngStyle.gt-lg": "ngStyle.gt-lg" }, usesInheritance: true, ngImport: i0 });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: DefaultStyleDirective, decorators: [{
            type: Directive,
            args: [{ selector, inputs }]
        }] });
/** Build a styles map from a list of styles, while sanitizing bad values first */
function buildMapFromList(styles, sanitize) {
    const sanitizeValue = (it) => {
        if (sanitize) {
            it.value = sanitize(it.value);
        }
        return it;
    };
    return styles
        .map(stringToKeyValue)
        .filter(entry => !!entry)
        .map(sanitizeValue)
        .reduce(keyValuesToMap, {});
}
//# sourceMappingURL=data:application/json;base64,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