/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** NgStyle allowed inputs */
export class NgStyleKeyValue {
    constructor(key, value, noQuotes = true) {
        this.key = key;
        this.value = value;
        this.key = noQuotes ? key.replace(/['"]/g, '').trim() : key.trim();
        this.value = noQuotes ? value.replace(/['"]/g, '').trim() : value.trim();
        this.value = this.value.replace(/;/, '');
    }
}
export function getType(target) {
    let what = typeof target;
    if (what === 'object') {
        return (target.constructor === Array) ? 'array' :
            (target.constructor === Set) ? 'set' : 'object';
    }
    return what;
}
/**
 * Split string of key:value pairs into Array of k-v pairs
 * e.g.  'key:value; key:value; key:value;' -> ['key:value',...]
 */
export function buildRawList(source, delimiter = ';') {
    return String(source)
        .trim()
        .split(delimiter)
        .map((val) => val.trim())
        .filter(val => val !== '');
}
/** Convert array of key:value strings to a iterable map object */
export function buildMapFromList(styles, sanitize) {
    const sanitizeValue = (it) => {
        if (sanitize) {
            it.value = sanitize(it.value);
        }
        return it;
    };
    return styles
        .map(stringToKeyValue)
        .filter(entry => !!entry)
        .map(sanitizeValue)
        .reduce(keyValuesToMap, {});
}
/** Convert Set<string> or raw Object to an iterable NgStyleMap */
export function buildMapFromSet(source, sanitize) {
    let list = [];
    if (getType(source) === 'set') {
        source.forEach(entry => list.push(entry));
    }
    else {
        Object.keys(source).forEach((key) => {
            list.push(`${key}:${source[key]}`);
        });
    }
    return buildMapFromList(list, sanitize);
}
/** Convert 'key:value' -> [key, value] */
export function stringToKeyValue(it) {
    const [key, ...vals] = it.split(':');
    return new NgStyleKeyValue(key, vals.join(':'));
}
/** Convert [ [key,value] ] -> { key : value } */
export function keyValuesToMap(map, entry) {
    if (!!entry.key) {
        map[entry.key] = entry.value;
    }
    return map;
}
//# sourceMappingURL=data:application/json;base64,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