/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Inject, Injectable, PLATFORM_ID } from '@angular/core';
import { isPlatformBrowser, isPlatformServer } from '@angular/common';
import { applyCssPrefixes } from '@angular/flex-layout/_private-utils';
import { SERVER_TOKEN } from '../tokens/server-token';
import { LAYOUT_CONFIG } from '../tokens/library-config';
import * as i0 from "@angular/core";
import * as i1 from "../stylesheet-map/stylesheet-map";
export class StyleUtils {
    constructor(_serverStylesheet, _serverModuleLoaded, _platformId, layoutConfig) {
        this._serverStylesheet = _serverStylesheet;
        this._serverModuleLoaded = _serverModuleLoaded;
        this._platformId = _platformId;
        this.layoutConfig = layoutConfig;
    }
    /**
     * Applies styles given via string pair or object map to the directive element
     */
    applyStyleToElement(element, style, value = null) {
        let styles = {};
        if (typeof style === 'string') {
            styles[style] = value;
            style = styles;
        }
        styles = this.layoutConfig.disableVendorPrefixes ? style : applyCssPrefixes(style);
        this._applyMultiValueStyleToElement(styles, element);
    }
    /**
     * Applies styles given via string pair or object map to the directive's element
     */
    applyStyleToElements(style, elements = []) {
        const styles = this.layoutConfig.disableVendorPrefixes ? style : applyCssPrefixes(style);
        elements.forEach(el => {
            this._applyMultiValueStyleToElement(styles, el);
        });
    }
    /**
     * Determine the DOM element's Flexbox flow (flex-direction)
     *
     * Check inline style first then check computed (stylesheet) style
     */
    getFlowDirection(target) {
        const query = 'flex-direction';
        let value = this.lookupStyle(target, query);
        const hasInlineValue = this.lookupInlineStyle(target, query) ||
            (isPlatformServer(this._platformId) && this._serverModuleLoaded) ? value : '';
        return [value || 'row', hasInlineValue];
    }
    hasWrap(target) {
        const query = 'flex-wrap';
        return this.lookupStyle(target, query) === 'wrap';
    }
    /**
     * Find the DOM element's raw attribute value (if any)
     */
    lookupAttributeValue(element, attribute) {
        return element.getAttribute(attribute) ?? '';
    }
    /**
     * Find the DOM element's inline style value (if any)
     */
    lookupInlineStyle(element, styleName) {
        return isPlatformBrowser(this._platformId) ?
            element.style.getPropertyValue(styleName) : getServerStyle(element, styleName);
    }
    /**
     * Determine the inline or inherited CSS style
     * NOTE: platform-server has no implementation for getComputedStyle
     */
    lookupStyle(element, styleName, inlineOnly = false) {
        let value = '';
        if (element) {
            let immediateValue = value = this.lookupInlineStyle(element, styleName);
            if (!immediateValue) {
                if (isPlatformBrowser(this._platformId)) {
                    if (!inlineOnly) {
                        value = getComputedStyle(element).getPropertyValue(styleName);
                    }
                }
                else {
                    if (this._serverModuleLoaded) {
                        value = this._serverStylesheet.getStyleForElement(element, styleName);
                    }
                }
            }
        }
        // Note: 'inline' is the default of all elements, unless UA stylesheet overrides;
        //       in which case getComputedStyle() should determine a valid value.
        return value ? value.trim() : '';
    }
    /**
     * Applies the styles to the element. The styles object map may contain an array of values
     * Each value will be added as element style
     * Keys are sorted to add prefixed styles (like -webkit-x) first, before the standard ones
     */
    _applyMultiValueStyleToElement(styles, element) {
        Object.keys(styles).sort().forEach(key => {
            const el = styles[key];
            const values = Array.isArray(el) ? el : [el];
            values.sort();
            for (let value of values) {
                value = value ? value + '' : '';
                if (isPlatformBrowser(this._platformId) || !this._serverModuleLoaded) {
                    isPlatformBrowser(this._platformId) ?
                        element.style.setProperty(key, value) : setServerStyle(element, key, value);
                }
                else {
                    this._serverStylesheet.addStyleToElement(element, key, value);
                }
            }
        });
    }
}
StyleUtils.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: StyleUtils, deps: [{ token: i1.StylesheetMap }, { token: SERVER_TOKEN }, { token: PLATFORM_ID }, { token: LAYOUT_CONFIG }], target: i0.ɵɵFactoryTarget.Injectable });
StyleUtils.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: StyleUtils, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: StyleUtils, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: function () { return [{ type: i1.StylesheetMap }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [SERVER_TOKEN]
                }] }, { type: Object, decorators: [{
                    type: Inject,
                    args: [PLATFORM_ID]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [LAYOUT_CONFIG]
                }] }]; } });
function getServerStyle(element, styleName) {
    const styleMap = readStyleAttribute(element);
    return styleMap[styleName] ?? '';
}
function setServerStyle(element, styleName, styleValue) {
    styleName = styleName.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
    const styleMap = readStyleAttribute(element);
    styleMap[styleName] = styleValue ?? '';
    writeStyleAttribute(element, styleMap);
}
function writeStyleAttribute(element, styleMap) {
    let styleAttrValue = '';
    for (const key in styleMap) {
        const newValue = styleMap[key];
        if (newValue) {
            styleAttrValue += `${key}:${styleMap[key]};`;
        }
    }
    element.setAttribute('style', styleAttrValue);
}
function readStyleAttribute(element) {
    const styleMap = {};
    const styleAttribute = element.getAttribute('style');
    if (styleAttribute) {
        const styleList = styleAttribute.split(/;+/g);
        for (let i = 0; i < styleList.length; i++) {
            const style = styleList[i].trim();
            if (style.length > 0) {
                const colonIndex = style.indexOf(':');
                if (colonIndex === -1) {
                    throw new Error(`Invalid CSS style: ${style}`);
                }
                const name = style.substr(0, colonIndex).trim();
                styleMap[name] = style.substr(colonIndex + 1).trim();
            }
        }
    }
    return styleMap;
}
//# sourceMappingURL=data:application/json;base64,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