/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Required as the signals library is in a separate package, so we need to explicitly ensure the
// global `ngDevMode` type is defined.
import '../../util/ng_dev_mode';
import { global } from '../../util/global';
/**
 * A `WeakRef`-compatible reference that fakes the API with a strong reference
 * internally.
 */
class LeakyRef {
    constructor(ref) {
        this.ref = ref;
    }
    deref() {
        return this.ref;
    }
}
// `WeakRef` is not always defined in every TS environment where Angular is compiled. Instead,
// read it off of the global context if available.
// tslint:disable-next-line: no-toplevel-property-access
let WeakRefImpl = global['WeakRef'] ?? LeakyRef;
export function newWeakRef(value) {
    if (typeof ngDevMode !== 'undefined' && ngDevMode && WeakRefImpl === undefined) {
        throw new Error(`Angular requires a browser which supports the 'WeakRef' API`);
    }
    return new WeakRefImpl(value);
}
export function setAlternateWeakRefImpl(impl) {
    // no-op since the alternate impl is included by default by the framework. Remove once internal
    // migration is complete.
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2Vha19yZWYuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9jb3JlL3NyYy9zaWduYWxzL3NyYy93ZWFrX3JlZi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFFSCxnR0FBZ0c7QUFDaEcsc0NBQXNDO0FBQ3RDLE9BQU8sd0JBQXdCLENBQUM7QUFFaEMsT0FBTyxFQUFDLE1BQU0sRUFBQyxNQUFNLG1CQUFtQixDQUFDO0FBRXpDOzs7R0FHRztBQUNILE1BQU0sUUFBUTtJQUNaLFlBQTZCLEdBQU07UUFBTixRQUFHLEdBQUgsR0FBRyxDQUFHO0lBQUcsQ0FBQztJQUV2QyxLQUFLO1FBQ0gsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDO0lBQ2xCLENBQUM7Q0FDRjtBQUVELDhGQUE4RjtBQUM5RixrREFBa0Q7QUFDbEQsd0RBQXdEO0FBQ3hELElBQUksV0FBVyxHQUEwQixNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksUUFBUSxDQUFDO0FBTXZFLE1BQU0sVUFBVSxVQUFVLENBQW1CLEtBQVE7SUFDbkQsSUFBSSxPQUFPLFNBQVMsS0FBSyxXQUFXLElBQUksU0FBUyxJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7UUFDOUUsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO0tBQ2hGO0lBQ0QsT0FBTyxJQUFJLFdBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNqQyxDQUFDO0FBTUQsTUFBTSxVQUFVLHVCQUF1QixDQUFDLElBQWlCO0lBQ3ZELCtGQUErRjtJQUMvRix5QkFBeUI7QUFDM0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG4vLyBSZXF1aXJlZCBhcyB0aGUgc2lnbmFscyBsaWJyYXJ5IGlzIGluIGEgc2VwYXJhdGUgcGFja2FnZSwgc28gd2UgbmVlZCB0byBleHBsaWNpdGx5IGVuc3VyZSB0aGVcbi8vIGdsb2JhbCBgbmdEZXZNb2RlYCB0eXBlIGlzIGRlZmluZWQuXG5pbXBvcnQgJy4uLy4uL3V0aWwvbmdfZGV2X21vZGUnO1xuXG5pbXBvcnQge2dsb2JhbH0gZnJvbSAnLi4vLi4vdXRpbC9nbG9iYWwnO1xuXG4vKipcbiAqIEEgYFdlYWtSZWZgLWNvbXBhdGlibGUgcmVmZXJlbmNlIHRoYXQgZmFrZXMgdGhlIEFQSSB3aXRoIGEgc3Ryb25nIHJlZmVyZW5jZVxuICogaW50ZXJuYWxseS5cbiAqL1xuY2xhc3MgTGVha3lSZWY8VD4vKiBpbXBsZW1lbnRzIFdlYWtSZWY8VD4gKi8ge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHJlZjogVCkge31cblxuICBkZXJlZigpOiBUfHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMucmVmO1xuICB9XG59XG5cbi8vIGBXZWFrUmVmYCBpcyBub3QgYWx3YXlzIGRlZmluZWQgaW4gZXZlcnkgVFMgZW52aXJvbm1lbnQgd2hlcmUgQW5ndWxhciBpcyBjb21waWxlZC4gSW5zdGVhZCxcbi8vIHJlYWQgaXQgb2ZmIG9mIHRoZSBnbG9iYWwgY29udGV4dCBpZiBhdmFpbGFibGUuXG4vLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6IG5vLXRvcGxldmVsLXByb3BlcnR5LWFjY2Vzc1xubGV0IFdlYWtSZWZJbXBsOiBXZWFrUmVmQ3Rvcnx1bmRlZmluZWQgPSBnbG9iYWxbJ1dlYWtSZWYnXSA/PyBMZWFreVJlZjtcblxuZXhwb3J0IGludGVyZmFjZSBXZWFrUmVmPFQgZXh0ZW5kcyBvYmplY3Q+IHtcbiAgZGVyZWYoKTogVHx1bmRlZmluZWQ7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBuZXdXZWFrUmVmPFQgZXh0ZW5kcyBvYmplY3Q+KHZhbHVlOiBUKTogV2Vha1JlZjxUPiB7XG4gIGlmICh0eXBlb2YgbmdEZXZNb2RlICE9PSAndW5kZWZpbmVkJyAmJiBuZ0Rldk1vZGUgJiYgV2Vha1JlZkltcGwgPT09IHVuZGVmaW5lZCkge1xuICAgIHRocm93IG5ldyBFcnJvcihgQW5ndWxhciByZXF1aXJlcyBhIGJyb3dzZXIgd2hpY2ggc3VwcG9ydHMgdGhlICdXZWFrUmVmJyBBUElgKTtcbiAgfVxuICByZXR1cm4gbmV3IFdlYWtSZWZJbXBsISh2YWx1ZSk7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgV2Vha1JlZkN0b3Ige1xuICBuZXc8VCBleHRlbmRzIG9iamVjdD4odmFsdWU6IFQpOiBXZWFrUmVmPFQ+O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gc2V0QWx0ZXJuYXRlV2Vha1JlZkltcGwoaW1wbDogV2Vha1JlZkN0b3IpIHtcbiAgLy8gbm8tb3Agc2luY2UgdGhlIGFsdGVybmF0ZSBpbXBsIGlzIGluY2x1ZGVkIGJ5IGRlZmF1bHQgYnkgdGhlIGZyYW1ld29yay4gUmVtb3ZlIG9uY2UgaW50ZXJuYWxcbiAgLy8gbWlncmF0aW9uIGlzIGNvbXBsZXRlLlxufVxuIl19