/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/**
 * Symbol used to tell `Signal`s apart from other functions.
 *
 * This can be used to auto-unwrap signals in various cases, or to auto-wrap non-signal values.
 */
const SIGNAL = Symbol('SIGNAL');
/**
 * Checks if the given `value` is a reactive `Signal`.
 *
 * @developerPreview
 */
export function isSignal(value) {
    return typeof value === 'function' && value[SIGNAL] !== undefined;
}
/**
 * Converts `fn` into a marked signal function (where `isSignal(fn)` will be `true`), and
 * potentially add some set of extra properties (passed as an object record `extraApi`).
 */
export function createSignalFromFunction(node, fn, extraApi = {}) {
    fn[SIGNAL] = node;
    // Copy properties from `extraApi` to `fn` to complete the desired API of the `Signal`.
    return Object.assign(fn, extraApi);
}
/**
 * The default equality function used for `signal` and `computed`, which treats objects and arrays
 * as never equal, and all other primitive values using identity semantics.
 *
 * This allows signals to hold non-primitive values (arrays, objects, other collections) and still
 * propagate change notification upon explicit mutation without identity change.
 *
 * @developerPreview
 */
export function defaultEquals(a, b) {
    // `Object.is` compares two values using identity semantics which is desired behavior for
    // primitive values. If `Object.is` determines two values to be equal we need to make sure that
    // those don't represent objects (we want to make sure that 2 objects are always considered
    // "unequal"). The null check is needed for the special case of JavaScript reporting null values
    // as objects (`typeof null === 'object'`).
    return (a === null || typeof a !== 'object') && Object.is(a, b);
}
//# sourceMappingURL=data:application/json;base64,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