/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext } from '../../di/contextual';
import { Injector } from '../../di/injector';
import { inject } from '../../di/injector_compatibility';
import { ɵɵdefineInjectable } from '../../di/interface/defs';
import { DestroyRef } from '../../linker/destroy_ref';
import { Watch } from '../../signals';
/**
 * Tracks all effects registered within a given application and runs them via `flush`.
 */
class EffectManager {
    constructor() {
        this.all = new Set();
        this.queue = new Map();
    }
    create(effectFn, destroyRef, allowSignalWrites) {
        const zone = (typeof Zone === 'undefined') ? null : Zone.current;
        const watch = new Watch(effectFn, (watch) => {
            if (!this.all.has(watch)) {
                return;
            }
            this.queue.set(watch, zone);
        }, allowSignalWrites);
        this.all.add(watch);
        // Effects start dirty.
        watch.notify();
        let unregisterOnDestroy;
        const destroy = () => {
            watch.cleanup();
            unregisterOnDestroy?.();
            this.all.delete(watch);
            this.queue.delete(watch);
        };
        unregisterOnDestroy = destroyRef?.onDestroy(destroy);
        return {
            destroy,
        };
    }
    flush() {
        if (this.queue.size === 0) {
            return;
        }
        for (const [watch, zone] of this.queue) {
            this.queue.delete(watch);
            if (zone) {
                zone.run(() => watch.run());
            }
            else {
                watch.run();
            }
        }
    }
    get isQueueEmpty() {
        return this.queue.size === 0;
    }
    /** @nocollapse */
    static { this.ɵprov = ɵɵdefineInjectable({
        token: EffectManager,
        providedIn: 'root',
        factory: () => new EffectManager(),
    }); }
}
export { EffectManager };
/**
 * Create a global `Effect` for the given reactive function.
 *
 * @developerPreview
 */
export function effect(effectFn, options) {
    !options?.injector && assertInInjectionContext(effect);
    const injector = options?.injector ?? inject(Injector);
    const effectManager = injector.get(EffectManager);
    const destroyRef = options?.manualCleanup !== true ? injector.get(DestroyRef) : null;
    return effectManager.create(effectFn, destroyRef, !!options?.allowSignalWrites);
}
//# sourceMappingURL=data:application/json;base64,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