/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { formatRuntimeError } from '../errors';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
import { stringifyCSSSelectorList } from './node_selector_matcher';
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyComponent {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const baseDef = getNgDirectiveDef(componentDefinition);
        const def = {
            ...baseDef,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            template: componentDefinition.template,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            dependencies: baseDef.standalone && componentDefinition.dependencies || null,
            getStandaloneInjector: null,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
            id: '',
        };
        initFeatures(def);
        const dependencies = componentDefinition.dependencies;
        def.directiveDefs = extractDefListOrFactory(dependencies, /* pipeDef */ false);
        def.pipeDefs = extractDefListOrFactory(dependencies, /* pipeDef */ true);
        def.id = getComponentId(def);
        return def;
    });
}
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = extractDefListOrFactory(directives, /* pipeDef */ false);
    def.pipeDefs = extractDefListOrFactory(pipes, /* pipeDef */ true);
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        return res;
    });
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵdefineDirective(directiveDefinition) {
    return noSideEffects(() => {
        const def = getNgDirectiveDef(directiveDefinition);
        initFeatures(def);
        return def;
    });
}
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
/**
 * Checks whether a given Component, Directive or Pipe is marked as standalone.
 * This will return false if passed anything other than a Component, Directive, or Pipe class
 * See [this guide](/guide/standalone-components) for additional information:
 *
 * @param type A reference to a Component, Directive or Pipe.
 * @publicApi
 */
export function isStandalone(type) {
    const def = getComponentDef(type) || getDirectiveDef(type) || getPipeDef(type);
    return def !== null ? def.standalone : false;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
function getNgDirectiveDef(directiveDefinition) {
    const declaredInputs = {};
    return {
        type: directiveDefinition.type,
        providersResolver: null,
        factory: null,
        hostBindings: directiveDefinition.hostBindings || null,
        hostVars: directiveDefinition.hostVars || 0,
        hostAttrs: directiveDefinition.hostAttrs || null,
        contentQueries: directiveDefinition.contentQueries || null,
        declaredInputs,
        exportAs: directiveDefinition.exportAs || null,
        standalone: directiveDefinition.standalone === true,
        selectors: directiveDefinition.selectors || EMPTY_ARRAY,
        viewQuery: directiveDefinition.viewQuery || null,
        features: directiveDefinition.features || null,
        setInput: null,
        findHostDirectiveDefs: null,
        hostDirectives: null,
        inputs: invertObject(directiveDefinition.inputs, declaredInputs),
        outputs: invertObject(directiveDefinition.outputs),
    };
}
function initFeatures(definition) {
    definition.features?.forEach((fn) => fn(definition));
}
function extractDefListOrFactory(dependencies, pipeDef) {
    if (!dependencies) {
        return null;
    }
    const defExtractor = pipeDef ? getPipeDef : extractDirectiveDef;
    return () => (typeof dependencies === 'function' ? dependencies() : dependencies)
        .map(dep => defExtractor(dep))
        .filter(nonNull);
}
/**
 * A map that contains the generated component IDs and type.
 */
export const GENERATED_COMP_IDS = new Map();
/**
 * A method can returns a component ID from the component definition using a variant of DJB2 hash
 * algorithm.
 */
function getComponentId(componentDef) {
    let hash = 0;
    // We cannot rely solely on the component selector as the same selector can be used in different
    // modules.
    //
    // `componentDef.style` is not used, due to it causing inconsistencies. Ex: when server
    // component styles has no sourcemaps and browsers do.
    //
    // Example:
    // https://github.com/angular/components/blob/d9f82c8f95309e77a6d82fd574c65871e91354c2/src/material/core/option/option.ts#L248
    // https://github.com/angular/components/blob/285f46dc2b4c5b127d356cb7c4714b221f03ce50/src/material/legacy-core/option/option.ts#L32
    const hashSelectors = [
        componentDef.selectors,
        componentDef.ngContentSelectors,
        componentDef.hostVars,
        componentDef.hostAttrs,
        componentDef.consts,
        componentDef.vars,
        componentDef.decls,
        componentDef.encapsulation,
        componentDef.standalone,
        componentDef.exportAs,
        JSON.stringify(componentDef.inputs),
        JSON.stringify(componentDef.outputs),
        // We cannot use 'componentDef.type.name' as the name of the symbol will change and will not
        // match in the server and browser bundles.
        Object.getOwnPropertyNames(componentDef.type.prototype),
        !!componentDef.contentQueries,
        !!componentDef.viewQuery,
    ].join('|');
    for (const char of hashSelectors) {
        hash = Math.imul(31, hash) + char.charCodeAt(0) << 0;
    }
    // Force positive number hash.
    // 2147483647 = equivalent of Integer.MAX_VALUE.
    hash += 2147483647 + 1;
    const compId = 'c' + hash;
    if (typeof ngDevMode === 'undefined' || ngDevMode) {
        if (GENERATED_COMP_IDS.has(compId)) {
            const previousCompDefType = GENERATED_COMP_IDS.get(compId);
            if (previousCompDefType !== componentDef.type) {
                console.warn(formatRuntimeError(-912 /* RuntimeErrorCode.COMPONENT_ID_COLLISION */, `Component ID generation collision detected. Components '${previousCompDefType.name}' and '${componentDef.type.name}' with selector '${stringifyCSSSelectorList(componentDef
                    .selectors)}' generated the same component ID. To fix this, you can change the selector of one of those components or add an extra host attribute to force a different ID.`));
            }
        }
        else {
            GENERATED_COMP_IDS.set(compId, componentDef.type);
        }
    }
    return compId;
}
//# sourceMappingURL=data:application/json;base64,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