/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertDefined, assertEqual, assertNumber, throwError } from '../util/assert';
import { getComponentDef, getNgModuleDef } from './definition';
import { isLContainer, isLView } from './interfaces/type_checks';
import { DECLARATION_COMPONENT_VIEW, HEADER_OFFSET, T_HOST, TVIEW } from './interfaces/view';
// [Assert functions do not constraint type when they are guarded by a truthy
// expression.](https://github.com/microsoft/TypeScript/issues/37295)
export function assertTNodeForLView(tNode, lView) {
    assertTNodeForTView(tNode, lView[TVIEW]);
}
export function assertTNodeForTView(tNode, tView) {
    assertTNode(tNode);
    tNode.hasOwnProperty('tView_') &&
        assertEqual(tNode.tView_, tView, 'This TNode does not belong to this TView.');
}
export function assertTNode(tNode) {
    assertDefined(tNode, 'TNode must be defined');
    if (!(tNode && typeof tNode === 'object' && tNode.hasOwnProperty('directiveStylingLast'))) {
        throwError('Not of type TNode, got: ' + tNode);
    }
}
export function assertTIcu(tIcu) {
    assertDefined(tIcu, 'Expected TIcu to be defined');
    if (!(typeof tIcu.currentCaseLViewIndex === 'number')) {
        throwError('Object is not of TIcu type.');
    }
}
export function assertComponentType(actual, msg = 'Type passed in is not ComponentType, it does not have \'ɵcmp\' property.') {
    if (!getComponentDef(actual)) {
        throwError(msg);
    }
}
export function assertNgModuleType(actual, msg = 'Type passed in is not NgModuleType, it does not have \'ɵmod\' property.') {
    if (!getNgModuleDef(actual)) {
        throwError(msg);
    }
}
export function assertCurrentTNodeIsParent(isParent) {
    assertEqual(isParent, true, 'currentTNode should be a parent');
}
export function assertHasParent(tNode) {
    assertDefined(tNode, 'currentTNode should exist!');
    assertDefined(tNode.parent, 'currentTNode should have a parent');
}
export function assertLContainer(value) {
    assertDefined(value, 'LContainer must be defined');
    assertEqual(isLContainer(value), true, 'Expecting LContainer');
}
export function assertLViewOrUndefined(value) {
    value && assertEqual(isLView(value), true, 'Expecting LView or undefined or null');
}
export function assertLView(value) {
    assertDefined(value, 'LView must be defined');
    assertEqual(isLView(value), true, 'Expecting LView');
}
export function assertFirstCreatePass(tView, errMessage) {
    assertEqual(tView.firstCreatePass, true, errMessage || 'Should only be called in first create pass.');
}
export function assertFirstUpdatePass(tView, errMessage) {
    assertEqual(tView.firstUpdatePass, true, errMessage || 'Should only be called in first update pass.');
}
/**
 * This is a basic sanity check that an object is probably a directive def. DirectiveDef is
 * an interface, so we can't do a direct instanceof check.
 */
export function assertDirectiveDef(obj) {
    if (obj.type === undefined || obj.selectors == undefined || obj.inputs === undefined) {
        throwError(`Expected a DirectiveDef/ComponentDef and this object does not seem to have the expected shape.`);
    }
}
export function assertIndexInDeclRange(lView, index) {
    const tView = lView[1];
    assertBetween(HEADER_OFFSET, tView.bindingStartIndex, index);
}
export function assertIndexInExpandoRange(lView, index) {
    const tView = lView[1];
    assertBetween(tView.expandoStartIndex, lView.length, index);
}
export function assertBetween(lower, upper, index) {
    if (!(lower <= index && index < upper)) {
        throwError(`Index out of range (expecting ${lower} <= ${index} < ${upper})`);
    }
}
export function assertProjectionSlots(lView, errMessage) {
    assertDefined(lView[DECLARATION_COMPONENT_VIEW], 'Component views should exist.');
    assertDefined(lView[DECLARATION_COMPONENT_VIEW][T_HOST].projection, errMessage ||
        'Components with projection nodes (<ng-content>) must have projection slots defined.');
}
export function assertParentView(lView, errMessage) {
    assertDefined(lView, errMessage || 'Component views should always have a parent view (component\'s host view)');
}
/**
 * This is a basic sanity check that the `injectorIndex` seems to point to what looks like a
 * NodeInjector data structure.
 *
 * @param lView `LView` which should be checked.
 * @param injectorIndex index into the `LView` where the `NodeInjector` is expected.
 */
export function assertNodeInjector(lView, injectorIndex) {
    assertIndexInExpandoRange(lView, injectorIndex);
    assertIndexInExpandoRange(lView, injectorIndex + 8 /* NodeInjectorOffset.PARENT */);
    assertNumber(lView[injectorIndex + 0], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 1], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 2], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 3], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 4], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 5], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 6], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 7], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 8 /* NodeInjectorOffset.PARENT */], 'injectorIndex should point to parent injector');
}
//# sourceMappingURL=data:application/json;base64,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