/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent, compileDirective } from '../render3/jit/directive';
import { compilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => compileDirective(type, meta));
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', (c = {}) => ({ changeDetection: ChangeDetectionStrategy.Default, ...c }), Directive, undefined, (type, meta) => compileComponent(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', (p) => ({ pure: true, ...p }), undefined, undefined, (type, meta) => compilePipe(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', (arg) => {
    if (!arg) {
        return {};
    }
    return typeof arg === 'string' ? { alias: arg } : arg;
});
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', (alias) => ({ alias }));
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *   }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for `Enter` key-press
 * events on the global `window`.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed enter {{counter}} number of times!</h1> Press enter key
 * to increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown.enter', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 * The list of valid key names for `keydown` and `keyup` events
 * can be found here:
 * https://www.w3.org/TR/DOM-Level-3-Events-key/#named-key-attribute-values
 *
 * Note that keys can also be combined, e.g. `@HostListener('keydown.shift.a')`.
 *
 * The global target names that can be used to prefix an event name are
 * `document:`, `window:` and `body:`.
 *
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));
//# sourceMappingURL=data:application/json;base64,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