/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { makePropDecorator } from '../util/decorators';
// Stores the default value of `emitDistinctChangesOnly` when the `emitDistinctChangesOnly` is not
// explicitly set.
export const emitDistinctChangesOnlyDefaultValue = true;
/**
 * Base class for query metadata.
 *
 * @see `ContentChildren`.
 * @see `ContentChild`.
 * @see `ViewChildren`.
 * @see `ViewChild`.
 *
 * @publicApi
 */
export class Query {
}
/**
 * ContentChildren decorator and metadata.
 *
 *
 * @Annotation
 * @publicApi
 */
export const ContentChildren = makePropDecorator('ContentChildren', (selector, data = {}) => ({
    selector,
    first: false,
    isViewQuery: false,
    descendants: false,
    emitDistinctChangesOnly: emitDistinctChangesOnlyDefaultValue,
    ...data
}), Query);
/**
 * ContentChild decorator and metadata.
 *
 *
 * @Annotation
 *
 * @publicApi
 */
export const ContentChild = makePropDecorator('ContentChild', (selector, data = {}) => ({ selector, first: true, isViewQuery: false, descendants: true, ...data }), Query);
/**
 * ViewChildren decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const ViewChildren = makePropDecorator('ViewChildren', (selector, data = {}) => ({
    selector,
    first: false,
    isViewQuery: true,
    descendants: true,
    emitDistinctChangesOnly: emitDistinctChangesOnlyDefaultValue,
    ...data
}), Query);
/**
 * ViewChild decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const ViewChild = makePropDecorator('ViewChild', (selector, data) => ({ selector, first: true, isViewQuery: true, descendants: true, ...data }), Query);
//# sourceMappingURL=data:application/json;base64,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