/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertLContainer } from '../render3/assert';
import { renderView } from '../render3/instructions/render';
import { createLView } from '../render3/instructions/shared';
import { DECLARATION_LCONTAINER, QUERIES } from '../render3/interfaces/view';
import { getCurrentTNode, getLView } from '../render3/state';
import { ViewRef as R3_ViewRef } from '../render3/view_ref';
import { assertDefined } from '../util/assert';
import { createElementRef } from './element_ref';
/**
 * Represents an embedded template that can be used to instantiate embedded views.
 * To instantiate embedded views based on a template, use the `ViewContainerRef`
 * method `createEmbeddedView()`.
 *
 * Access a `TemplateRef` instance by placing a directive on an `<ng-template>`
 * element (or directive prefixed with `*`). The `TemplateRef` for the embedded view
 * is injected into the constructor of the directive,
 * using the `TemplateRef` token.
 *
 * You can also use a `Query` to find a `TemplateRef` associated with
 * a component or a directive.
 *
 * @see `ViewContainerRef`
 * @see [Navigate the Component Tree with DI](guide/dependency-injection-navtree)
 *
 * @publicApi
 */
class TemplateRef {
    /**
     * @internal
     * @nocollapse
     */
    static { this.__NG_ELEMENT_ID__ = injectTemplateRef; }
}
export { TemplateRef };
const ViewEngineTemplateRef = TemplateRef;
// TODO(alxhub): combine interface and implementation. Currently this is challenging since something
// in g3 depends on them being separate.
const R3TemplateRef = class TemplateRef extends ViewEngineTemplateRef {
    constructor(_declarationLView, _declarationTContainer, elementRef) {
        super();
        this._declarationLView = _declarationLView;
        this._declarationTContainer = _declarationTContainer;
        this.elementRef = elementRef;
    }
    /**
     * Returns an `ssrId` associated with a TView, which was used to
     * create this instance of the `TemplateRef`.
     *
     * @internal
     */
    get ssrId() {
        return this._declarationTContainer.tView?.ssrId || null;
    }
    createEmbeddedView(context, injector) {
        return this.createEmbeddedViewImpl(context, injector, null);
    }
    /**
     * @internal
     */
    createEmbeddedViewImpl(context, injector, hydrationInfo) {
        const embeddedTView = this._declarationTContainer.tView;
        const embeddedLView = createLView(this._declarationLView, embeddedTView, context, 16 /* LViewFlags.CheckAlways */, null, embeddedTView.declTNode, null, null, null, injector || null, hydrationInfo || null);
        const declarationLContainer = this._declarationLView[this._declarationTContainer.index];
        ngDevMode && assertLContainer(declarationLContainer);
        embeddedLView[DECLARATION_LCONTAINER] = declarationLContainer;
        const declarationViewLQueries = this._declarationLView[QUERIES];
        if (declarationViewLQueries !== null) {
            embeddedLView[QUERIES] = declarationViewLQueries.createEmbeddedView(embeddedTView);
        }
        renderView(embeddedTView, embeddedLView, context);
        return new R3_ViewRef(embeddedLView);
    }
};
/**
 * Creates a TemplateRef given a node.
 *
 * @returns The TemplateRef instance to use
 */
export function injectTemplateRef() {
    return createTemplateRef(getCurrentTNode(), getLView());
}
/**
 * Creates a TemplateRef and stores it on the injector.
 *
 * @param hostTNode The node on which a TemplateRef is requested
 * @param hostLView The `LView` to which the node belongs
 * @returns The TemplateRef instance or null if we can't create a TemplateRef on a given node type
 */
export function createTemplateRef(hostTNode, hostLView) {
    if (hostTNode.type & 4 /* TNodeType.Container */) {
        ngDevMode && assertDefined(hostTNode.tView, 'TView must be allocated');
        return new R3TemplateRef(hostLView, hostTNode, createElementRef(hostTNode, hostLView));
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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