/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime
 * logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    // We also prepend `0` to non-compile-time errors.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message : ''}`;
    if (ngDevMode && code < 0) {
        const addPeriodSeparator = !errorMessage.match(/[.,;!?\n]$/);
        const separator = addPeriodSeparator ? '.' : '';
        errorMessage =
            `${errorMessage}${separator} Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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