/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import '../util/ng_dev_mode';
import { RuntimeError } from '../errors';
import { stringify } from '../util/stringify';
import { resolveForwardRef } from './forward_ref';
import { getInjectImplementation, injectRootLimpMode } from './inject_switch';
import { InjectFlags } from './interface/injector';
const _THROW_IF_NOT_FOUND = {};
export const THROW_IF_NOT_FOUND = _THROW_IF_NOT_FOUND;
/*
 * Name of a property (that we patch onto DI decorator), which is used as an annotation of which
 * InjectFlag this decorator represents. This allows to avoid direct references to the DI decorators
 * in the code, thus making them tree-shakable.
 */
const DI_DECORATOR_FLAG = '__NG_DI_FLAG__';
export const NG_TEMP_TOKEN_PATH = 'ngTempTokenPath';
const NG_TOKEN_PATH = 'ngTokenPath';
const NEW_LINE = /\n/gm;
const NO_NEW_LINE = 'ɵ';
export const SOURCE = '__source';
/**
 * Current injector value used by `inject`.
 * - `undefined`: it is an error to call `inject`
 * - `null`: `inject` can be called but there is no injector (limp-mode).
 * - Injector instance: Use the injector for resolution.
 */
let _currentInjector = undefined;
export function getCurrentInjector() {
    return _currentInjector;
}
export function setCurrentInjector(injector) {
    const former = _currentInjector;
    _currentInjector = injector;
    return former;
}
export function injectInjectorOnly(token, flags = InjectFlags.Default) {
    if (_currentInjector === undefined) {
        throw new RuntimeError(-203 /* RuntimeErrorCode.MISSING_INJECTION_CONTEXT */, ngDevMode &&
            `inject() must be called from an injection context such as a constructor, a factory function, a field initializer, or a function used with \`runInInjectionContext\`.`);
    }
    else if (_currentInjector === null) {
        return injectRootLimpMode(token, undefined, flags);
    }
    else {
        return _currentInjector.get(token, flags & InjectFlags.Optional ? null : undefined, flags);
    }
}
export function ɵɵinject(token, flags = InjectFlags.Default) {
    return (getInjectImplementation() || injectInjectorOnly)(resolveForwardRef(token), flags);
}
/**
 * Throws an error indicating that a factory function could not be generated by the compiler for a
 * particular class.
 *
 * The name of the class is not mentioned here, but will be in the generated factory function name
 * and thus in the stack trace.
 *
 * @codeGenApi
 */
export function ɵɵinvalidFactoryDep(index) {
    throw new RuntimeError(202 /* RuntimeErrorCode.INVALID_FACTORY_DEPENDENCY */, ngDevMode &&
        `This constructor is not compatible with Angular Dependency Injection because its dependency at index ${index} of the parameter list is invalid.
This can happen if the dependency type is a primitive like a string or if an ancestor of this class is missing an Angular decorator.

Please check that 1) the type for the parameter at index ${index} is correct and 2) the correct Angular decorators are defined for this class and its ancestors.`);
}
/**
 * Injects a token from the currently active injector.
 * `inject` is only supported during instantiation of a dependency by the DI system. It can be used
 * during:
 * - Construction (via the `constructor`) of a class being instantiated by the DI system, such
 * as an `@Injectable` or `@Component`.
 * - In the initializer for fields of such classes.
 * - In the factory function specified for `useFactory` of a `Provider` or an `@Injectable`.
 * - In the `factory` function specified for an `InjectionToken`.
 *
 * @param token A token that represents a dependency that should be injected.
 * @param flags Optional flags that control how injection is executed.
 * The flags correspond to injection strategies that can be specified with
 * parameter decorators `@Host`, `@Self`, `@SkipSelf`, and `@Optional`.
 * @returns the injected value if operation is successful, `null` otherwise.
 * @throws if called outside of a supported context.
 *
 * @usageNotes
 * In practice the `inject()` calls are allowed in a constructor, a constructor parameter and a
 * field initializer:
 *
 * ```typescript
 * @Injectable({providedIn: 'root'})
 * export class Car {
 *   radio: Radio|undefined;
 *   // OK: field initializer
 *   spareTyre = inject(Tyre);
 *
 *   constructor() {
 *     // OK: constructor body
 *     this.radio = inject(Radio);
 *   }
 * }
 * ```
 *
 * It is also legal to call `inject` from a provider's factory:
 *
 * ```typescript
 * providers: [
 *   {provide: Car, useFactory: () => {
 *     // OK: a class factory
 *     const engine = inject(Engine);
 *     return new Car(engine);
 *   }}
 * ]
 * ```
 *
 * Calls to the `inject()` function outside of the class creation context will result in error. Most
 * notably, calls to `inject()` are disallowed after a class instance was created, in methods
 * (including lifecycle hooks):
 *
 * ```typescript
 * @Component({ ... })
 * export class CarComponent {
 *   ngOnInit() {
 *     // ERROR: too late, the component instance was already created
 *     const engine = inject(Engine);
 *     engine.start();
 *   }
 * }
 * ```
 *
 * @publicApi
 */
export function inject(token, flags = InjectFlags.Default) {
    return ɵɵinject(token, convertToBitFlags(flags));
}
// Converts object-based DI flags (`InjectOptions`) to bit flags (`InjectFlags`).
export function convertToBitFlags(flags) {
    if (typeof flags === 'undefined' || typeof flags === 'number') {
        return flags;
    }
    // While TypeScript doesn't accept it without a cast, bitwise OR with false-y values in
    // JavaScript is a no-op. We can use that for a very codesize-efficient conversion from
    // `InjectOptions` to `InjectFlags`.
    return (0 /* InternalInjectFlags.Default */ | // comment to force a line break in the formatter
        (flags.optional && 8 /* InternalInjectFlags.Optional */) |
        (flags.host && 1 /* InternalInjectFlags.Host */) |
        (flags.self && 2 /* InternalInjectFlags.Self */) |
        (flags.skipSelf && 4 /* InternalInjectFlags.SkipSelf */));
}
export function injectArgs(types) {
    const args = [];
    for (let i = 0; i < types.length; i++) {
        const arg = resolveForwardRef(types[i]);
        if (Array.isArray(arg)) {
            if (arg.length === 0) {
                throw new RuntimeError(900 /* RuntimeErrorCode.INVALID_DIFFER_INPUT */, ngDevMode && 'Arguments array must have arguments.');
            }
            let type = undefined;
            let flags = InjectFlags.Default;
            for (let j = 0; j < arg.length; j++) {
                const meta = arg[j];
                const flag = getInjectFlag(meta);
                if (typeof flag === 'number') {
                    // Special case when we handle @Inject decorator.
                    if (flag === -1 /* DecoratorFlags.Inject */) {
                        type = meta.token;
                    }
                    else {
                        flags |= flag;
                    }
                }
                else {
                    type = meta;
                }
            }
            args.push(ɵɵinject(type, flags));
        }
        else {
            args.push(ɵɵinject(arg));
        }
    }
    return args;
}
/**
 * Attaches a given InjectFlag to a given decorator using monkey-patching.
 * Since DI decorators can be used in providers `deps` array (when provider is configured using
 * `useFactory`) without initialization (e.g. `Host`) and as an instance (e.g. `new Host()`), we
 * attach the flag to make it available both as a static property and as a field on decorator
 * instance.
 *
 * @param decorator Provided DI decorator.
 * @param flag InjectFlag that should be applied.
 */
export function attachInjectFlag(decorator, flag) {
    decorator[DI_DECORATOR_FLAG] = flag;
    decorator.prototype[DI_DECORATOR_FLAG] = flag;
    return decorator;
}
/**
 * Reads monkey-patched property that contains InjectFlag attached to a decorator.
 *
 * @param token Token that may contain monkey-patched DI flags property.
 */
export function getInjectFlag(token) {
    return token[DI_DECORATOR_FLAG];
}
export function catchInjectorError(e, token, injectorErrorName, source) {
    const tokenPath = e[NG_TEMP_TOKEN_PATH];
    if (token[SOURCE]) {
        tokenPath.unshift(token[SOURCE]);
    }
    e.message = formatError('\n' + e.message, tokenPath, injectorErrorName, source);
    e[NG_TOKEN_PATH] = tokenPath;
    e[NG_TEMP_TOKEN_PATH] = null;
    throw e;
}
export function formatError(text, obj, injectorErrorName, source = null) {
    text = text && text.charAt(0) === '\n' && text.charAt(1) == NO_NEW_LINE ? text.slice(2) : text;
    let context = stringify(obj);
    if (Array.isArray(obj)) {
        context = obj.map(stringify).join(' -> ');
    }
    else if (typeof obj === 'object') {
        let parts = [];
        for (let key in obj) {
            if (obj.hasOwnProperty(key)) {
                let value = obj[key];
                parts.push(key + ':' + (typeof value === 'string' ? JSON.stringify(value) : stringify(value)));
            }
        }
        context = `{${parts.join(', ')}}`;
    }
    return `${injectorErrorName}${source ? '(' + source + ')' : ''}[${context}]: ${text.replace(NEW_LINE, '\n  ')}`;
}
//# sourceMappingURL=data:application/json;base64,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