/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { RuntimeError } from '../errors';
import { getCurrentInjector, setCurrentInjector } from './injector_compatibility';
import { getInjectImplementation, setInjectImplementation } from './inject_switch';
import { R3Injector } from './r3_injector';
/**
 * Runs the given function in the context of the given `Injector`.
 *
 * Within the function's stack frame, `inject` can be used to inject dependencies from the given
 * `Injector`. Note that `inject` is only usable synchronously, and cannot be used in any
 * asynchronous callbacks or after any `await` points.
 *
 * @param injector the injector which will satisfy calls to `inject` while `fn` is executing
 * @param fn the closure to be run in the context of `injector`
 * @returns the return value of the function, if any
 * @publicApi
 */
export function runInInjectionContext(injector, fn) {
    if (injector instanceof R3Injector) {
        injector.assertNotDestroyed();
    }
    const prevInjector = setCurrentInjector(injector);
    const previousInjectImplementation = setInjectImplementation(undefined);
    try {
        return fn();
    }
    finally {
        setCurrentInjector(prevInjector);
        setInjectImplementation(previousInjectImplementation);
    }
}
/**
 * Asserts that the current stack frame is within an injection context and has access to `inject`.
 *
 * @param debugFn a reference to the function making the assertion (used for the error message).
 *
 * @publicApi
 */
export function assertInInjectionContext(debugFn) {
    // Taking a `Function` instead of a string name here prevents the unminified name of the function
    // from being retained in the bundle regardless of minification.
    if (!getInjectImplementation() && !getCurrentInjector()) {
        throw new RuntimeError(-203 /* RuntimeErrorCode.MISSING_INJECTION_CONTEXT */, ngDevMode &&
            (debugFn.name +
                '() can only be used within an injection context such as a constructor, a factory function, a field initializer, or a function used with `runInInjectionContext`'));
    }
}
//# sourceMappingURL=data:application/json;base64,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