/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { isComponentHost } from '../render3/interfaces/type_checks';
import { DECLARATION_COMPONENT_VIEW } from '../render3/interfaces/view';
import { getCurrentTNode, getLView } from '../render3/state';
import { getComponentLViewByIndex } from '../render3/util/view_utils';
import { ViewRef } from '../render3/view_ref';
/**
 * Base class that provides change detection functionality.
 * A change-detection tree collects all views that are to be checked for changes.
 * Use the methods to add and remove views from the tree, initiate change-detection,
 * and explicitly mark views as _dirty_, meaning that they have changed and need to be re-rendered.
 *
 * @see [Using change detection hooks](guide/lifecycle-hooks#using-change-detection-hooks)
 * @see [Defining custom change detection](guide/lifecycle-hooks#defining-custom-change-detection)
 *
 * @usageNotes
 *
 * The following examples demonstrate how to modify default change-detection behavior
 * to perform explicit detection when needed.
 *
 * ### Use `markForCheck()` with `CheckOnce` strategy
 *
 * The following example sets the `OnPush` change-detection strategy for a component
 * (`CheckOnce`, rather than the default `CheckAlways`), then forces a second check
 * after an interval.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts"
 * region="mark-for-check"></code-example>
 *
 * ### Detach change detector to limit how often check occurs
 *
 * The following example defines a component with a large list of read-only data
 * that is expected to change constantly, many times per second.
 * To improve performance, we want to check and update the list
 * less often than the changes actually occur. To do that, we detach
 * the component's change detector and perform an explicit local check every five seconds.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="detach"></code-example>
 *
 *
 * ### Reattaching a detached component
 *
 * The following example creates a component displaying live data.
 * The component detaches its change detector from the main change detector tree
 * when the `live` property is set to false, and reattaches it when the property
 * becomes true.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="reattach"></code-example>
 *
 * @publicApi
 */
class ChangeDetectorRef {
    /**
     * @internal
     * @nocollapse
     */
    static { this.__NG_ELEMENT_ID__ = injectChangeDetectorRef; }
}
export { ChangeDetectorRef };
/** Returns a ChangeDetectorRef (a.k.a. a ViewRef) */
export function injectChangeDetectorRef(flags) {
    return createViewRef(getCurrentTNode(), getLView(), (flags & 16 /* InternalInjectFlags.ForPipe */) === 16 /* InternalInjectFlags.ForPipe */);
}
/**
 * Creates a ViewRef and stores it on the injector as ChangeDetectorRef (public alias).
 *
 * @param tNode The node that is requesting a ChangeDetectorRef
 * @param lView The view to which the node belongs
 * @param isPipe Whether the view is being injected into a pipe.
 * @returns The ChangeDetectorRef to use
 */
function createViewRef(tNode, lView, isPipe) {
    if (isComponentHost(tNode) && !isPipe) {
        // The LView represents the location where the component is declared.
        // Instead we want the LView for the component View and so we need to look it up.
        const componentView = getComponentLViewByIndex(tNode.index, lView); // look down
        return new ViewRef(componentView, componentView);
    }
    else if (tNode.type & (3 /* TNodeType.AnyRNode */ | 12 /* TNodeType.AnyContainer */ | 32 /* TNodeType.Icu */)) {
        // The LView represents the location where the injection is requested from.
        // We need to locate the containing LView (in case where the `lView` is an embedded view)
        const hostComponentView = lView[DECLARATION_COMPONENT_VIEW]; // look up
        return new ViewRef(hostComponentView, lView);
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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