/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext, computed, DestroyRef, inject, signal } from '@angular/core';
import { RuntimeError } from '../../src/errors';
import { untracked } from '../../src/signals';
export function toSignal(source, options) {
    const requiresCleanup = !options?.manualCleanup;
    requiresCleanup && !options?.injector && assertInInjectionContext(toSignal);
    const cleanupRef = requiresCleanup ? options?.injector?.get(DestroyRef) ?? inject(DestroyRef) : null;
    // Note: T is the Observable value type, and U is the initial value type. They don't have to be
    // the same - the returned signal gives values of type `T`.
    let state;
    if (options?.requireSync) {
        // Initially the signal is in a `NoValue` state.
        state = signal({ kind: 0 /* StateKind.NoValue */ });
    }
    else {
        // If an initial value was passed, use it. Otherwise, use `undefined` as the initial value.
        state = signal({ kind: 1 /* StateKind.Value */, value: options?.initialValue });
    }
    const sub = source.subscribe({
        next: value => state.set({ kind: 1 /* StateKind.Value */, value }),
        error: error => state.set({ kind: 2 /* StateKind.Error */, error }),
        // Completion of the Observable is meaningless to the signal. Signals don't have a concept of
        // "complete".
    });
    if (ngDevMode && options?.requireSync && untracked(state).kind === 0 /* StateKind.NoValue */) {
        throw new RuntimeError(601 /* RuntimeErrorCode.REQUIRE_SYNC_WITHOUT_SYNC_EMIT */, '`toSignal()` called with `requireSync` but `Observable` did not emit synchronously.');
    }
    // Unsubscribe when the current context is destroyed, if requested.
    cleanupRef?.onDestroy(sub.unsubscribe.bind(sub));
    // The actual returned signal is a `computed` of the `State` signal, which maps the various states
    // to either values or errors.
    return computed(() => {
        const current = state();
        switch (current.kind) {
            case 1 /* StateKind.Value */:
                return current.value;
            case 2 /* StateKind.Error */:
                throw current.error;
            case 0 /* StateKind.NoValue */:
                // This shouldn't really happen because the error is thrown on creation.
                // TODO(alxhub): use a RuntimeError when we finalize the error semantics
                throw new RuntimeError(601 /* RuntimeErrorCode.REQUIRE_SYNC_WITHOUT_SYNC_EMIT */, '`toSignal()` called with `requireSync` but `Observable` did not emit synchronously.');
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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