/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext, DestroyRef, effect, inject, Injector, untracked } from '@angular/core';
import { ReplaySubject } from 'rxjs';
/**
 * Exposes the value of an Angular `Signal` as an RxJS `Observable`.
 *
 * The signal's value will be propagated into the `Observable`'s subscribers using an `effect`.
 *
 * `toObservable` must be called in an injection context unless an injector is provided via options.
 *
 * @developerPreview
 */
export function toObservable(source, options) {
    !options?.injector && assertInInjectionContext(toObservable);
    const injector = options?.injector ?? inject(Injector);
    const subject = new ReplaySubject(1);
    const watcher = effect(() => {
        let value;
        try {
            value = source();
        }
        catch (err) {
            untracked(() => subject.error(err));
            return;
        }
        untracked(() => subject.next(value));
    }, { injector, manualCleanup: true });
    injector.get(DestroyRef).onDestroy(() => {
        watcher.destroy();
        subject.complete();
    });
    return subject.asObservable();
}
//# sourceMappingURL=data:application/json;base64,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