/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as ir from '../../ir';
/**
 * Optimize variables declared and used in the IR.
 *
 * Variables are eagerly generated by pipeline stages for all possible values that could be
 * referenced. This stage processes the list of declared variables and all variable usages,
 * and optimizes where possible. It performs 3 main optimizations:
 *
 *   * It transforms variable declarations to side effectful expressions when the
 *     variable is not used, but its initializer has global effects which other
 *     operations rely upon.
 *   * It removes variable declarations if those variables are not referenced and
 *     either they do not have global effects, or nothing relies on them.
 *   * It inlines variable declarations when those variables are only used once
 *     and the inlining is semantically safe.
 *
 * To guarantee correctness, analysis of "fences" in the instruction lists is used to determine
 * which optimizations are safe to perform.
 */
export function phaseVariableOptimization(cpl, options) {
    for (const [_, view] of cpl.views) {
        optimizeVariablesInOpList(view.create, options);
        optimizeVariablesInOpList(view.update, options);
        for (const op of view.create) {
            if (op.kind === ir.OpKind.Listener) {
                optimizeVariablesInOpList(op.handlerOps, options);
            }
        }
    }
}
/**
 * A [fence](https://en.wikipedia.org/wiki/Memory_barrier) flag for an expression which indicates
 * how that expression can be optimized in relation to other expressions or instructions.
 *
 * `Fence`s are a bitfield, so multiple flags may be set on a single expression.
 */
var Fence;
(function (Fence) {
    /**
     * Empty flag (no fence exists).
     */
    Fence[Fence["None"] = 0] = "None";
    /**
     * A context read fence, meaning that the expression in question reads from the "current view"
     * context of the runtime.
     */
    Fence[Fence["ViewContextRead"] = 1] = "ViewContextRead";
    /**
     * A context write fence, meaning that the expression in question writes to the "current view"
     * context of the runtime.
     *
     * Note that all `ContextWrite` fences are implicitly `ContextRead` fences as operations which
     * change the view context do so based on the current one.
     */
    Fence[Fence["ViewContextWrite"] = 3] = "ViewContextWrite";
    /**
     * Indicates that a call is required for its side-effects, even if nothing reads its result.
     *
     * This is also true of `ViewContextWrite` operations **if** they are followed by a
     * `ViewContextRead`.
     */
    Fence[Fence["SideEffectful"] = 4] = "SideEffectful";
})(Fence || (Fence = {}));
/**
 * Process a list of operations and optimize variables within that list.
 */
function optimizeVariablesInOpList(ops, options) {
    const varDecls = new Map();
    const varUsages = new Map();
    // Track variables that are used outside of the immediate operation list. For example, within
    // `ListenerOp` handler operations of listeners in the current operation list.
    const varRemoteUsages = new Set();
    const opMap = new Map();
    // First, extract information about variables declared or used within the whole list.
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable) {
            if (varDecls.has(op.xref) || varUsages.has(op.xref)) {
                throw new Error(`Should not see two declarations of the same variable: ${op.xref}`);
            }
            varDecls.set(op.xref, op);
            varUsages.set(op.xref, 0);
        }
        opMap.set(op, collectOpInfo(op));
        countVariableUsages(op, varUsages, varRemoteUsages);
    }
    // The next step is to remove any variable declarations for variables that aren't used. The
    // variable initializer expressions may be side-effectful, so they may need to be retained as
    // expression statements.
    // Track whether we've seen an operation which reads from the view context yet. This is used to
    // determine whether a write to the view context in a variable initializer can be observed.
    let contextIsUsed = false;
    // Note that iteration through the list happens in reverse, which guarantees that we'll process
    // all reads of a variable prior to processing its declaration.
    for (const op of ops.reversed()) {
        const opInfo = opMap.get(op);
        if (op.kind === ir.OpKind.Variable && varUsages.get(op.xref) === 0) {
            // This variable is unused and can be removed. We might need to keep the initializer around,
            // though, if something depends on it running.
            if ((contextIsUsed && opInfo.fences & Fence.ViewContextWrite) ||
                (opInfo.fences & Fence.SideEffectful)) {
                // This variable initializer has a side effect which must be retained. Either:
                //  * it writes to the view context, and we know there is a future operation which depends
                //    on that write, or
                //  * it's an operation which is inherently side-effectful.
                // We can't remove the initializer, but we can remove the variable declaration itself and
                // replace it with a side-effectful statement.
                const stmtOp = ir.createStatementOp(op.initializer.toStmt());
                opMap.set(stmtOp, opInfo);
                ir.OpList.replace(op, stmtOp);
            }
            else {
                // It's safe to delete this entire variable declaration as nothing depends on it, even
                // side-effectfully. Note that doing this might make other variables unused. Since we're
                // iterating in reverse order, we should always be processing usages before declarations
                // and therefore by the time we get to a declaration, all removable usages will have been
                // removed.
                uncountVariableUsages(op, varUsages);
                ir.OpList.remove(op);
            }
            opMap.delete(op);
            varDecls.delete(op.xref);
            varUsages.delete(op.xref);
            continue;
        }
        // Does this operation depend on the view context?
        if (opInfo.fences & Fence.ViewContextRead) {
            contextIsUsed = true;
        }
    }
    // Next, inline any remaining variables with exactly one usage.
    const toInline = [];
    for (const [id, count] of varUsages) {
        // We can inline variables that:
        //  - are used once
        //  - are not used remotely
        if (count !== 1) {
            // We can't inline this variable as it's used more than once.
            continue;
        }
        if (varRemoteUsages.has(id)) {
            // This variable is used once, but across an operation boundary, so it can't be inlined.
            continue;
        }
        toInline.push(id);
    }
    let candidate;
    while (candidate = toInline.pop()) {
        // We will attempt to inline this variable. If inlining fails (due to fences for example),
        // no future operation will make inlining legal.
        const decl = varDecls.get(candidate);
        const varInfo = opMap.get(decl);
        // Scan operations following the variable declaration and look for the point where that variable
        // is used. There should only be one usage given the precondition above.
        for (let targetOp = decl.next; targetOp.kind !== ir.OpKind.ListEnd; targetOp = targetOp.next) {
            const opInfo = opMap.get(targetOp);
            // Is the variable used in this operation?
            if (opInfo.variablesUsed.has(candidate)) {
                if (options.conservative && !allowConservativeInlining(decl, targetOp)) {
                    // We're in conservative mode, and this variable is not eligible for inlining into the
                    // target operation in this mode.
                    break;
                }
                // Yes, try to inline it. Inlining may not be successful if fences in this operation before
                // the variable's usage cannot be safely crossed.
                if (tryInlineVariableInitializer(candidate, decl.initializer, targetOp, varInfo.fences)) {
                    // Inlining was successful! Update the tracking structures to reflect the inlined
                    // variable.
                    opInfo.variablesUsed.delete(candidate);
                    // Add all variables used in the variable's initializer to its new usage site.
                    for (const id of varInfo.variablesUsed) {
                        opInfo.variablesUsed.add(id);
                    }
                    // Merge fences in the variable's initializer into its new usage site.
                    opInfo.fences |= varInfo.fences;
                    // Delete tracking info related to the declaration.
                    varDecls.delete(candidate);
                    varUsages.delete(candidate);
                    opMap.delete(decl);
                    // And finally, delete the original declaration from the operation list.
                    ir.OpList.remove(decl);
                }
                // Whether inlining succeeded or failed, we're done processing this variable.
                break;
            }
            // If the variable is not used in this operation, then we'd need to inline across it. Check if
            // that's safe to do.
            if (!safeToInlinePastFences(opInfo.fences, varInfo.fences)) {
                // We can't safely inline this variable beyond this operation, so don't proceed with
                // inlining this variable.
                break;
            }
        }
    }
}
/**
 * Given an `ir.Expression`, returns the `Fence` flags for that expression type.
 */
function fencesForIrExpression(expr) {
    switch (expr.kind) {
        case ir.ExpressionKind.NextContext:
            return Fence.ViewContextWrite;
        case ir.ExpressionKind.RestoreView:
            return Fence.ViewContextWrite | Fence.SideEffectful;
        case ir.ExpressionKind.Reference:
            return Fence.ViewContextRead;
        default:
            return Fence.None;
    }
}
/**
 * Build the `OpInfo` structure for the given `op`. This performs two operations:
 *
 *  * It tracks which variables are used in the operation's expressions.
 *  * It rolls up fence flags for expressions within the operation.
 */
function collectOpInfo(op) {
    let fences = Fence.None;
    const variablesUsed = new Set();
    ir.visitExpressionsInOp(op, expr => {
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                variablesUsed.add(expr.xref);
                break;
            default:
                fences |= fencesForIrExpression(expr);
        }
    });
    return { fences, variablesUsed };
}
/**
 * Count the number of usages of each variable, being careful to track whether those usages are
 * local or remote.
 */
function countVariableUsages(op, varUsages, varRemoteUsage) {
    ir.visitExpressionsInOp(op, (expr, flags) => {
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        varUsages.set(expr.xref, count + 1);
        if (flags & ir.VisitorContextFlag.InChildOperation) {
            varRemoteUsage.add(expr.xref);
        }
    });
}
/**
 * Remove usages of a variable in `op` from the `varUsages` tracking.
 */
function uncountVariableUsages(op, varUsages) {
    ir.visitExpressionsInOp(op, expr => {
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        else if (count === 0) {
            throw new Error(`Inaccurate variable count: ${expr.xref} - found another read but count is already 0`);
        }
        varUsages.set(expr.xref, count - 1);
    });
}
/**
 * Checks whether it's safe to inline a variable across a particular operation.
 *
 * @param fences the fences of the operation which the inlining will cross
 * @param declFences the fences of the variable being inlined.
 */
function safeToInlinePastFences(fences, declFences) {
    if (fences & Fence.ViewContextWrite) {
        // It's not safe to inline context reads across context writes.
        if (declFences & Fence.ViewContextRead) {
            return false;
        }
    }
    else if (fences & Fence.ViewContextRead) {
        // It's not safe to inline context writes across context reads.
        if (declFences & Fence.ViewContextWrite) {
            return false;
        }
    }
    return true;
}
/**
 * Attempt to inline the initializer of a variable into a target operation's expressions.
 *
 * This may or may not be safe to do. For example, the variable could be read following the
 * execution of an expression with fences that don't permit the variable to be inlined across them.
 */
function tryInlineVariableInitializer(id, initializer, target, declFences) {
    // We use `ir.transformExpressionsInOp` to walk the expressions and inline the variable if
    // possible. Since this operation is callback-based, once inlining succeeds or fails we can't
    // "stop" the expression processing, and have to keep track of whether inlining has succeeded or
    // is no longer allowed.
    let inlined = false;
    let inliningAllowed = true;
    ir.transformExpressionsInOp(target, (expr, flags) => {
        if (inlined || !inliningAllowed) {
            // Either the inlining has already succeeded, or we've passed a fence that disallows inlining
            // at this point, so don't try.
            return expr;
        }
        else if ((flags & ir.VisitorContextFlag.InChildOperation) && (declFences & Fence.ViewContextRead)) {
            // We cannot inline variables that are sensitive to the current context across operation
            // boundaries.
            return expr;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                if (expr.xref === id) {
                    // This is the usage site of the variable. Since nothing has disallowed inlining, it's
                    // safe to inline the initializer here.
                    inlined = true;
                    return initializer;
                }
                break;
            default:
                // For other types of `ir.Expression`s, whether inlining is allowed depends on their fences.
                const exprFences = fencesForIrExpression(expr);
                inliningAllowed = inliningAllowed && safeToInlinePastFences(exprFences, declFences);
                break;
        }
        return expr;
    }, ir.VisitorContextFlag.None);
    return inlined;
}
/**
 * Determines whether inlining of `decl` should be allowed in "conservative" mode.
 *
 * In conservative mode, inlining behavior is limited to those operations which the
 * `TemplateDefinitionBuilder` supported, with the goal of producing equivalent output.
 */
function allowConservativeInlining(decl, target) {
    // TODO(alxhub): understand exactly how TemplateDefinitionBuilder approaches inlining, and record
    // that behavior here.
    switch (decl.variable.kind) {
        case ir.SemanticVariableKind.Identifier:
            return false;
        case ir.SemanticVariableKind.Context:
            // Context can only be inlined into other variables.
            return target.kind === ir.OpKind.Variable;
        default:
            return true;
    }
}
//# sourceMappingURL=data:application/json;base64,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