/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../src/output/output_ast';
import * as ir from '../ir';
import { phaseConstCollection } from './phases/const_collection';
import { phaseEmptyElements } from './phases/empty_elements';
import { phaseGenerateAdvance } from './phases/generate_advance';
import { phaseReify } from './phases/reify';
import { phaseSlotAllocation } from './phases/slot_allocation';
import { phaseVarCounting } from './phases/var_counting';
import { phaseNaming } from './phases/naming';
import { phaseLocalRefs } from './phases/local_refs';
import { phaseGenerateVariables } from './phases/generate_variables';
import { phaseResolveNames } from './phases/resolve_names';
import { phaseResolveContexts } from './phases/resolve_contexts';
import { phaseVariableOptimization } from './phases/variable_optimization';
import { phaseChaining } from './phases/chaining';
import { phaseMergeNextContext } from './phases/next_context_merging';
/**
 * Run all transformation phases in the correct order against a `ComponentCompilation`. After this
 * processing, the compilation should be in a state where it can be emitted via `emitTemplateFn`.s
 */
export function transformTemplate(cpl) {
    phaseGenerateVariables(cpl);
    phaseResolveNames(cpl);
    phaseResolveContexts(cpl);
    phaseLocalRefs(cpl);
    phaseEmptyElements(cpl);
    phaseConstCollection(cpl);
    phaseSlotAllocation(cpl);
    phaseVarCounting(cpl);
    phaseGenerateAdvance(cpl);
    phaseNaming(cpl);
    phaseVariableOptimization(cpl, { conservative: true });
    phaseMergeNextContext(cpl);
    phaseReify(cpl);
    phaseChaining(cpl);
}
/**
 * Compile all views in the given `ComponentCompilation` into the final template function, which may
 * reference constants defined in a `ConstantPool`.
 */
export function emitTemplateFn(tpl, pool) {
    const rootFn = emitView(tpl.root);
    emitChildViews(tpl.root, pool);
    return rootFn;
}
function emitChildViews(parent, pool) {
    for (const view of parent.tpl.views.values()) {
        if (view.parent !== parent.xref) {
            continue;
        }
        // Child views are emitted depth-first.
        emitChildViews(view, pool);
        const viewFn = emitView(view);
        pool.statements.push(viewFn.toDeclStmt(viewFn.name));
    }
}
/**
 * Emit a template function for an individual `ViewCompilation` (which may be either the root view
 * or an embedded view).
 */
function emitView(view) {
    if (view.fnName === null) {
        throw new Error(`AssertionError: view ${view.xref} is unnamed`);
    }
    const createStatements = [];
    for (const op of view.create) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all create ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        createStatements.push(op.statement);
    }
    const updateStatements = [];
    for (const op of view.update) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all update ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        updateStatements.push(op.statement);
    }
    const createCond = maybeGenerateRfBlock(1, createStatements);
    const updateCond = maybeGenerateRfBlock(2, updateStatements);
    return o.fn([
        new o.FnParam('rf'),
        new o.FnParam('ctx'),
    ], [
        ...createCond,
        ...updateCond,
    ], 
    /* type */ undefined, /* sourceSpan */ undefined, view.fnName);
}
function maybeGenerateRfBlock(flag, statements) {
    if (statements.length === 0) {
        return [];
    }
    return [
        o.ifStmt(new o.BinaryOperatorExpr(o.BinaryOperator.BitwiseAnd, o.variable('rf'), o.literal(flag)), statements),
    ];
}
//# sourceMappingURL=data:application/json;base64,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