"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createFormatDiagnosticHost = exports.createFileSystemCompilerHost = exports.FileSystemHost = void 0;
const ts = require("typescript");
/**
 * Implementation of a TypeScript parse config host that relies fully on
 * a given virtual file system.
 */
class FileSystemHost {
    _fileSystem;
    useCaseSensitiveFileNames = ts.sys.useCaseSensitiveFileNames;
    constructor(_fileSystem) {
        this._fileSystem = _fileSystem;
    }
    fileExists(path) {
        return this._fileSystem.fileExists(this._fileSystem.resolve(path));
    }
    readFile(path) {
        const content = this._fileSystem.read(this._fileSystem.resolve(path));
        if (content === null) {
            return undefined;
        }
        // Strip BOM as otherwise TSC methods (e.g. "getWidth") will return an offset which
        // which breaks the CLI UpdateRecorder. https://github.com/angular/angular/pull/30719
        return content.replace(/^\uFEFF/, '');
    }
    readDirectory(rootDir, extensions, excludes, includes, depth) {
        if (ts.matchFiles === undefined) {
            throw Error('Unable to read directory in virtual file system host. This means that ' +
                'TypeScript changed its file matching internals.\n\nPlease consider downgrading your ' +
                'TypeScript version, and report an issue in the Angular Components repository.');
        }
        return ts.matchFiles(rootDir, extensions, extensions, includes, this.useCaseSensitiveFileNames, '/', depth, p => this._getFileSystemEntries(p), p => this._fileSystem.resolve(p), p => this._fileSystem.directoryExists(this._fileSystem.resolve(p)));
    }
    _getFileSystemEntries(path) {
        return this._fileSystem.readDirectory(this._fileSystem.resolve(path));
    }
}
exports.FileSystemHost = FileSystemHost;
/**
 * Creates a TypeScript compiler host that fully relies fully on the given
 * virtual file system. i.e. no interactions with the working directory.
 */
function createFileSystemCompilerHost(options, fileSystem) {
    const host = ts.createCompilerHost(options, true);
    const virtualHost = new FileSystemHost(fileSystem);
    host.readFile = virtualHost.readFile.bind(virtualHost);
    host.readDirectory = virtualHost.readDirectory.bind(virtualHost);
    host.fileExists = virtualHost.fileExists.bind(virtualHost);
    host.directoryExists = dirPath => fileSystem.directoryExists(fileSystem.resolve(dirPath));
    host.getCurrentDirectory = () => '/';
    host.getCanonicalFileName = p => fileSystem.resolve(p);
    return host;
}
exports.createFileSystemCompilerHost = createFileSystemCompilerHost;
/** Creates a format diagnostic host that works with the given file system. */
function createFormatDiagnosticHost(fileSystem) {
    return {
        getCanonicalFileName: p => fileSystem.resolve(p),
        getCurrentDirectory: () => '/',
        getNewLine: () => '\n',
    };
}
exports.createFormatDiagnosticHost = createFormatDiagnosticHost;
//# sourceMappingURL=data:application/json;base64,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