/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ESCAPE, hasModifierKey } from '@angular/cdk/keycodes';
import { Subject } from 'rxjs';
/**
 * Reference to a dialog opened via the Dialog service.
 */
export class DialogRef {
    constructor(overlayRef, config) {
        this.overlayRef = overlayRef;
        this.config = config;
        /** Emits when the dialog has been closed. */
        this.closed = new Subject();
        this.disableClose = config.disableClose;
        this.backdropClick = overlayRef.backdropClick();
        this.keydownEvents = overlayRef.keydownEvents();
        this.outsidePointerEvents = overlayRef.outsidePointerEvents();
        this.id = config.id; // By the time the dialog is created we are guaranteed to have an ID.
        this.keydownEvents.subscribe(event => {
            if (event.keyCode === ESCAPE && !this.disableClose && !hasModifierKey(event)) {
                event.preventDefault();
                this.close(undefined, { focusOrigin: 'keyboard' });
            }
        });
        this.backdropClick.subscribe(() => {
            if (!this.disableClose) {
                this.close(undefined, { focusOrigin: 'mouse' });
            }
        });
        this._detachSubscription = overlayRef.detachments().subscribe(() => {
            // Check specifically for `false`, because we want `undefined` to be treated like `true`.
            if (config.closeOnOverlayDetachments !== false) {
                this.close();
            }
        });
    }
    /**
     * Close the dialog.
     * @param result Optional result to return to the dialog opener.
     * @param options Additional options to customize the closing behavior.
     */
    close(result, options) {
        if (this.containerInstance) {
            const closedSubject = this.closed;
            this.containerInstance._closeInteractionType = options?.focusOrigin || 'program';
            // Drop the detach subscription first since it can be triggered by the
            // `dispose` call and override the result of this closing sequence.
            this._detachSubscription.unsubscribe();
            this.overlayRef.dispose();
            closedSubject.next(result);
            closedSubject.complete();
            this.componentInstance = this.containerInstance = null;
        }
    }
    /** Updates the position of the dialog based on the current position strategy. */
    updatePosition() {
        this.overlayRef.updatePosition();
        return this;
    }
    /**
     * Updates the dialog's width and height.
     * @param width New width of the dialog.
     * @param height New height of the dialog.
     */
    updateSize(width = '', height = '') {
        this.overlayRef.updateSize({ width, height });
        return this;
    }
    /** Add a CSS class or an array of classes to the overlay pane. */
    addPanelClass(classes) {
        this.overlayRef.addPanelClass(classes);
        return this;
    }
    /** Remove a CSS class or an array of classes from the overlay pane. */
    removePanelClass(classes) {
        this.overlayRef.removePanelClass(classes);
        return this;
    }
}
//# sourceMappingURL=data:application/json;base64,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