import { Injectable, Optional, SkipSelf } from '@angular/core';
import { Subject } from 'rxjs';
import { NgxMatDateAdapter } from './core/date-adapter';
import * as i0 from "@angular/core";
import * as i1 from "./core/date-adapter";
/** A class representing a range of dates. */
export class NgxDateRange {
    constructor(
    /** The start date of the range. */
    start, 
    /** The end date of the range. */
    end) {
        this.start = start;
        this.end = end;
    }
}
/**
 * A selection model containing a date selection.
 * @docs-private
 */
export class NgxMatDateSelectionModel {
    constructor(
    /** The current selection. */
    selection, _adapter) {
        this.selection = selection;
        this._adapter = _adapter;
        this._selectionChanged = new Subject();
        /** Emits when the selection has changed. */
        this.selectionChanged = this._selectionChanged;
        this.selection = selection;
    }
    /**
     * Updates the current selection in the model.
     * @param value New selection that should be assigned.
     * @param source Object that triggered the selection change.
     */
    updateSelection(value, source) {
        const oldValue = this.selection;
        this.selection = value;
        this._selectionChanged.next({ selection: value, source, oldValue });
    }
    ngOnDestroy() {
        this._selectionChanged.complete();
    }
    _isValidDateInstance(date) {
        return this._adapter.isDateInstance(date) && this._adapter.isValid(date);
    }
}
/** @nocollapse */ NgxMatDateSelectionModel.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatDateSelectionModel, deps: "invalid", target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ NgxMatDateSelectionModel.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatDateSelectionModel });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatDateSelectionModel, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: undefined }, { type: i1.NgxMatDateAdapter }]; } });
/**
 * A selection model that contains a single date.
 * @docs-private
 */
export class NgxMatSingleDateSelectionModel extends NgxMatDateSelectionModel {
    constructor(adapter) {
        super(null, adapter);
    }
    /**
     * Adds a date to the current selection. In the case of a single date selection, the added date
     * simply overwrites the previous selection
     */
    add(date) {
        super.updateSelection(date, this);
    }
    /** Checks whether the current selection is valid. */
    isValid() {
        return this.selection != null && this._isValidDateInstance(this.selection);
    }
    /**
     * Checks whether the current selection is complete. In the case of a single date selection, this
     * is true if the current selection is not null.
     */
    isComplete() {
        return this.selection != null;
    }
    /** Clones the selection model. */
    clone() {
        const clone = new NgxMatSingleDateSelectionModel(this._adapter);
        clone.updateSelection(this.selection, this);
        return clone;
    }
}
/** @nocollapse */ NgxMatSingleDateSelectionModel.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatSingleDateSelectionModel, deps: [{ token: i1.NgxMatDateAdapter }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ NgxMatSingleDateSelectionModel.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatSingleDateSelectionModel });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatSingleDateSelectionModel, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i1.NgxMatDateAdapter }]; } });
/**
 * A selection model that contains a date range.
 * @docs-private
 */
export class NgxMatRangeDateSelectionModel extends NgxMatDateSelectionModel {
    constructor(adapter) {
        super(new NgxDateRange(null, null), adapter);
    }
    /**
     * Adds a date to the current selection. In the case of a date range selection, the added date
     * fills in the next `null` value in the range. If both the start and the end already have a date,
     * the selection is reset so that the given date is the new `start` and the `end` is null.
     */
    add(date) {
        let { start, end } = this.selection;
        if (start == null) {
            start = date;
        }
        else if (end == null) {
            end = date;
        }
        else {
            start = date;
            end = null;
        }
        super.updateSelection(new NgxDateRange(start, end), this);
    }
    /** Checks whether the current selection is valid. */
    isValid() {
        const { start, end } = this.selection;
        // Empty ranges are valid.
        if (start == null && end == null) {
            return true;
        }
        // Complete ranges are only valid if both dates are valid and the start is before the end.
        if (start != null && end != null) {
            return (this._isValidDateInstance(start) &&
                this._isValidDateInstance(end) &&
                this._adapter.compareDate(start, end) <= 0);
        }
        // Partial ranges are valid if the start/end is valid.
        return ((start == null || this._isValidDateInstance(start)) &&
            (end == null || this._isValidDateInstance(end)));
    }
    /**
     * Checks whether the current selection is complete. In the case of a date range selection, this
     * is true if the current selection has a non-null `start` and `end`.
     */
    isComplete() {
        return this.selection.start != null && this.selection.end != null;
    }
    /** Clones the selection model. */
    clone() {
        const clone = new NgxMatRangeDateSelectionModel(this._adapter);
        clone.updateSelection(this.selection, this);
        return clone;
    }
}
/** @nocollapse */ NgxMatRangeDateSelectionModel.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatRangeDateSelectionModel, deps: [{ token: i1.NgxMatDateAdapter }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ NgxMatRangeDateSelectionModel.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatRangeDateSelectionModel });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.0.2", ngImport: i0, type: NgxMatRangeDateSelectionModel, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i1.NgxMatDateAdapter }]; } });
/** @docs-private */
export function NGX_MAT_SINGLE_DATE_SELECTION_MODEL_FACTORY(parent, adapter) {
    return parent || new NgxMatSingleDateSelectionModel(adapter);
}
/**
 * Used to provide a single selection model to a component.
 * @docs-private
 */
export const NGX_MAT_SINGLE_DATE_SELECTION_MODEL_PROVIDER = {
    provide: NgxMatDateSelectionModel,
    deps: [[new Optional(), new SkipSelf(), NgxMatDateSelectionModel], NgxMatDateAdapter],
    useFactory: NGX_MAT_SINGLE_DATE_SELECTION_MODEL_FACTORY,
};
/** @docs-private */
export function NGX_MAT_RANGE_DATE_SELECTION_MODEL_FACTORY(parent, adapter) {
    return parent || new NgxMatRangeDateSelectionModel(adapter);
}
/**
 * Used to provide a range selection model to a component.
 * @docs-private
 */
export const NGX_MAT_RANGE_DATE_SELECTION_MODEL_PROVIDER = {
    provide: NgxMatDateSelectionModel,
    deps: [[new Optional(), new SkipSelf(), NgxMatDateSelectionModel], NgxMatDateAdapter],
    useFactory: NGX_MAT_RANGE_DATE_SELECTION_MODEL_FACTORY,
};
//# sourceMappingURL=data:application/json;base64,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