// This file contains the `_computeAriaAccessibleName` function, which computes what the *expected*
// ARIA accessible name would be for a given element. Implements a subset of ARIA specification
// [Accessible Name and Description Computation 1.2](https://www.w3.org/TR/accname-1.2/).
//
// Specification accname-1.2 can be summarized by returning the result of the first method
// available.
//
//  1. `aria-labelledby` attribute
//     ```
//       <!-- example using aria-labelledby-->
//       <label id='label-id'>Start Date</label>
//       <input aria-labelledby='label-id'/>
//     ```
//  2. `aria-label` attribute (e.g. `<input aria-label="Departure"/>`)
//  3. Label with `for`/`id`
//     ```
//       <!-- example using for/id -->
//       <label for="current-node">Label</label>
//       <input id="current-node"/>
//     ```
//  4. `placeholder` attribute (e.g. `<input placeholder="06/03/1990"/>`)
//  5. `title` attribute (e.g. `<input title="Check-In"/>`)
//  6. text content
//     ```
//       <!-- example using text content -->
//       <label for="current-node"><span>Departure</span> Date</label>
//       <input id="current-node"/>
//     ```
/**
 * Computes the *expected* ARIA accessible name for argument element based on [accname-1.2
 * specification](https://www.w3.org/TR/accname-1.2/). Implements a subset of accname-1.2,
 * and should only be used for the Datepicker's specific use case.
 *
 * Intended use:
 * This is not a general use implementation. Only implements the parts of accname-1.2 that are
 * required for the Datepicker's specific use case. This function is not intended for any other
 * use.
 *
 * Limitations:
 *  - Only covers the needs of `matStartDate` and `matEndDate`. Does not support other use cases.
 *  - See NOTES's in implementation for specific details on what parts of the accname-1.2
 *  specification are not implemented.
 *
 *  @param element {HTMLInputElement} native &lt;input/&gt; element of `matStartDate` or
 *  `matEndDate` component. Corresponds to the 'Root Element' from accname-1.2
 *
 *  @return expected ARIA accessible name of argument &lt;input/&gt;
 */
export function _computeAriaAccessibleName(element) {
    return _computeAriaAccessibleNameInternal(element, true);
}
/**
 * Determine if argument node is an Element based on `nodeType` property. This function is safe to
 * use with server-side rendering.
 */
function ssrSafeIsElement(node) {
    return node.nodeType === Node.ELEMENT_NODE;
}
/**
 * Determine if argument node is an HTMLInputElement based on `nodeName` property. This funciton is
 * safe to use with server-side rendering.
 */
function ssrSafeIsHTMLInputElement(node) {
    return node.nodeName === 'INPUT';
}
/**
 * Determine if argument node is an HTMLTextAreaElement based on `nodeName` property. This
 * funciton is safe to use with server-side rendering.
 */
function ssrSafeIsHTMLTextAreaElement(node) {
    return node.nodeName === 'TEXTAREA';
}
/**
 * Calculate the expected ARIA accessible name for given DOM Node. Given DOM Node may be either the
 * "Root node" passed to `_computeAriaAccessibleName` or "Current node" as result of recursion.
 *
 * @return the accessible name of argument DOM Node
 *
 * @param currentNode node to determine accessible name of
 * @param isDirectlyReferenced true if `currentNode` is the root node to calculate ARIA accessible
 * name of. False if it is a result of recursion.
 */
function _computeAriaAccessibleNameInternal(currentNode, isDirectlyReferenced) {
    // NOTE: this differs from accname-1.2 specification.
    //  - Does not implement Step 1. of accname-1.2: '''If `currentNode`'s role prohibits naming,
    //    return the empty string ("")'''.
    //  - Does not implement Step 2.A. of accname-1.2: '''if current node is hidden and not directly
    //    referenced by aria-labelledby... return the empty string.'''
    // acc-name-1.2 Step 2.B.: aria-labelledby
    if (ssrSafeIsElement(currentNode) && isDirectlyReferenced) {
        const labelledbyIds = currentNode.getAttribute?.('aria-labelledby')?.split(/\s+/g) || [];
        const validIdRefs = labelledbyIds.reduce((validIds, id) => {
            const elem = document.getElementById(id);
            if (elem) {
                validIds.push(elem);
            }
            return validIds;
        }, []);
        if (validIdRefs.length) {
            return validIdRefs
                .map(idRef => {
                return _computeAriaAccessibleNameInternal(idRef, false);
            })
                .join(' ');
        }
    }
    // acc-name-1.2 Step 2.C.: aria-label
    if (ssrSafeIsElement(currentNode)) {
        const ariaLabel = currentNode.getAttribute('aria-label')?.trim();
        if (ariaLabel) {
            return ariaLabel;
        }
    }
    // acc-name-1.2 Step 2.D. attribute or element that defines a text alternative
    //
    // NOTE: this differs from accname-1.2 specification.
    // Only implements Step 2.D. for `<label>`,`<input/>`, and `<textarea/>` element. Does not
    // implement other elements that have an attribute or element that defines a text alternative.
    if (ssrSafeIsHTMLInputElement(currentNode) || ssrSafeIsHTMLTextAreaElement(currentNode)) {
        // use label with a `for` attribute referencing the current node
        if (currentNode.labels?.length) {
            return Array.from(currentNode.labels)
                .map(x => _computeAriaAccessibleNameInternal(x, false))
                .join(' ');
        }
        // use placeholder if available
        const placeholder = currentNode.getAttribute('placeholder')?.trim();
        if (placeholder) {
            return placeholder;
        }
        // use title if available
        const title = currentNode.getAttribute('title')?.trim();
        if (title) {
            return title;
        }
    }
    // NOTE: this differs from accname-1.2 specification.
    //  - does not implement acc-name-1.2 Step 2.E.: '''if the current node is a control embedded
    //     within the label... then include the embedded control as part of the text alternative in
    //     the following manner...'''. Step 2E applies to embedded controls such as textbox, listbox,
    //     range, etc.
    //  - does not implement acc-name-1.2 step 2.F.: check that '''role allows name from content''',
    //    which applies to `currentNode` and its children.
    //  - does not implement acc-name-1.2 Step 2.F.ii.: '''Check for CSS generated textual content'''
    //    (e.g. :before and :after).
    //  - does not implement acc-name-1.2 Step 2.I.: '''if the current node has a Tooltip attribute,
    //    return its value'''
    // Return text content with whitespace collapsed into a single space character. Accomplish
    // acc-name-1.2 steps 2F, 2G, and 2H.
    return (currentNode.textContent || '').replace(/\s+/g, ' ').trim();
}
//# sourceMappingURL=data:application/json;base64,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