"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.isContentAction = exports.ActionList = exports.UnknownActionException = void 0;
const core_1 = require("@angular-devkit/core");
class UnknownActionException extends core_1.BaseException {
    constructor(action) {
        super(`Unknown action: "${action.kind}".`);
    }
}
exports.UnknownActionException = UnknownActionException;
let _id = 1;
class ActionList {
    constructor() {
        this._actions = [];
    }
    _action(action) {
        this._actions.push({
            ...action,
            id: _id++,
            parent: this._actions[this._actions.length - 1]?.id ?? 0,
        });
    }
    create(path, content) {
        this._action({ kind: 'c', path, content });
    }
    overwrite(path, content) {
        this._action({ kind: 'o', path, content });
    }
    rename(path, to) {
        this._action({ kind: 'r', path, to });
    }
    delete(path) {
        this._action({ kind: 'd', path });
    }
    optimize() {
        const toCreate = new Map();
        const toRename = new Map();
        const toOverwrite = new Map();
        const toDelete = new Set();
        for (const action of this._actions) {
            switch (action.kind) {
                case 'c':
                    toCreate.set(action.path, action.content);
                    break;
                case 'o':
                    if (toCreate.has(action.path)) {
                        toCreate.set(action.path, action.content);
                    }
                    else {
                        toOverwrite.set(action.path, action.content);
                    }
                    break;
                case 'd':
                    toDelete.add(action.path);
                    break;
                case 'r':
                    const maybeCreate = toCreate.get(action.path);
                    const maybeOverwrite = toOverwrite.get(action.path);
                    if (maybeCreate) {
                        toCreate.delete(action.path);
                        toCreate.set(action.to, maybeCreate);
                    }
                    if (maybeOverwrite) {
                        toOverwrite.delete(action.path);
                        toOverwrite.set(action.to, maybeOverwrite);
                    }
                    let maybeRename = undefined;
                    for (const [from, to] of toRename.entries()) {
                        if (to == action.path) {
                            maybeRename = from;
                            break;
                        }
                    }
                    if (maybeRename) {
                        toRename.set(maybeRename, action.to);
                    }
                    if (!maybeCreate && !maybeOverwrite && !maybeRename) {
                        toRename.set(action.path, action.to);
                    }
                    break;
            }
        }
        this._actions = [];
        toDelete.forEach((x) => {
            this.delete(x);
        });
        toRename.forEach((to, from) => {
            this.rename(from, to);
        });
        toCreate.forEach((content, path) => {
            this.create(path, content);
        });
        toOverwrite.forEach((content, path) => {
            this.overwrite(path, content);
        });
    }
    push(action) {
        this._actions.push(action);
    }
    get(i) {
        return this._actions[i];
    }
    has(action) {
        for (let i = 0; i < this._actions.length; i++) {
            const a = this._actions[i];
            if (a.id == action.id) {
                return true;
            }
            if (a.id > action.id) {
                return false;
            }
        }
        return false;
    }
    find(predicate) {
        return this._actions.find(predicate) || null;
    }
    forEach(fn, thisArg) {
        this._actions.forEach(fn, thisArg);
    }
    get length() {
        return this._actions.length;
    }
    [Symbol.iterator]() {
        return this._actions[Symbol.iterator]();
    }
}
exports.ActionList = ActionList;
function isContentAction(action) {
    return action.kind == 'c' || action.kind == 'o';
}
exports.isContentAction = isContentAction;
//# sourceMappingURL=data:application/json;base64,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