"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDispatcher = void 0;
const api_1 = require("./api");
const exception_1 = require("./exception");
/**
 * OnReady a dispatcher that can dispatch to a sub job, depending on conditions.
 * @param options
 */
function createDispatcher(options = {}) {
    let defaultDelegate = null;
    const conditionalDelegateList = [];
    const job = Object.assign((argument, context) => {
        const maybeDelegate = conditionalDelegateList.find(([predicate]) => predicate(argument));
        let delegate = null;
        if (maybeDelegate) {
            delegate = context.scheduler.schedule(maybeDelegate[1], argument);
        }
        else if (defaultDelegate) {
            delegate = context.scheduler.schedule(defaultDelegate, argument);
        }
        else {
            throw new exception_1.JobDoesNotExistException('<null>');
        }
        context.inboundBus.subscribe(delegate.inboundBus);
        return delegate.outboundBus;
    }, {
        jobDescription: options,
    });
    return Object.assign(job, {
        setDefaultJob(name) {
            if ((0, api_1.isJobHandler)(name)) {
                name = name.jobDescription.name === undefined ? null : name.jobDescription.name;
            }
            defaultDelegate = name;
        },
        addConditionalJob(predicate, name) {
            conditionalDelegateList.push([predicate, name]);
        },
        // TODO: Remove return-only generic from createDispatcher() API.
    });
}
exports.createDispatcher = createDispatcher;
//# sourceMappingURL=data:application/json;base64,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